/*
 *  This file is part of RawTherapee.
 *
 *  Copyright (c) 2004-2010 Gabor Horvath <hgabor@rawtherapee.com>
 *
 *  RawTherapee is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  RawTherapee is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with RawTherapee.  If not, see <https://www.gnu.org/licenses/>.
 *
 *  These median implementations from Flössie and Ingo Weyrich are inspired by this work:
 *
 *  http://ndevilla.free.fr/median/median.pdf
 *  http://pages.ripco.net/~jgamble/nw.html
 *  https://github.com/hoytech/Algorithm-Networksort-Chooser
 *
 *  Instead of using the PIX_SORT and PIX_SWAP macros we use std::min() and std::max()
 *  because it turned out that it generates much faster (branch free) code on machines which support SSE
 *
 */

#pragma once

#include <array>
#include <algorithm>

#include "opthelper.h"

#if defined __GNUC__ && __GNUC__>=6 && defined __SSE2__
    #pragma GCC diagnostic ignored "-Wignored-attributes"
#endif

template<typename T, std::size_t N>
inline T median(std::array<T, N> array)
{
    const typename std::array<T, N>::iterator middle = array.begin() + N / 2;
    std::nth_element(array.begin(), middle, array.end());

    return
        (N % 2)
            ? *middle
            : ((*middle + *std::max_element(array.begin(), middle)) / static_cast<T>(2));
}

template<typename T>
inline T median(std::array<T, 3> array)
{
    return std::max(std::min(array[0], array[1]), std::min(array[2], std::max(array[0], array[1])));
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 3> array)
{
    return vmaxf(vminf(array[0], array[1]), vminf(array[2], vmaxf(array[0], array[1])));
}
#endif


template<typename T>
inline T median(std::array<T, 4> array)
{
    float val1 = std::max(std::min(array[0], array[1]), std::min(array[2], array[3]));
    float val2 = std::min(std::max(array[0], array[1]), std::max(array[2], array[3]));
    return (val1 + val2) / 2.f;
}

template<typename T>
inline T median(std::array<T, 5> array)
{
    T tmp = std::min(array[0], array[1]);
    array[1] = std::max(array[0], array[1]);
    array[0] = tmp;
    tmp = std::min(array[3], array[4]);
    array[4] = std::max(array[3], array[4]);
    array[3] = std::max(array[0], tmp);
    array[1] = std::min(array[1], array[4]);
    tmp = std::min(array[1], array[2]);
    array[2] = std::max(array[1], array[2]);
    array[1] = tmp;
    tmp = std::min(array[2], array[3]);
    return std::max(array[1], tmp);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 5> array)
{
    vfloat tmp = vminf(array[0], array[1]);
    array[1] = vmaxf(array[0], array[1]);
    array[0] = tmp;
    tmp = vminf(array[3], array[4]);
    array[4] = vmaxf(array[3], array[4]);
    array[3] = vmaxf(array[0], tmp);
    array[1] = vminf(array[1], array[4]);
    tmp = vminf(array[1], array[2]);
    array[2] = vmaxf(array[1], array[2]);
    array[1] = tmp;
    tmp = vminf(array[2], array[3]);
    return vmaxf(array[1], tmp);
}
#endif

template<typename T>
inline T median(std::array<T, 7> array)
{
    T tmp = std::min(array[0], array[5]);
    array[5] = std::max(array[0], array[5]);
    array[0] = tmp;
    tmp = std::min(array[0], array[3]);
    array[3] = std::max(array[0], array[3]);
    array[0] = tmp;
    tmp = std::min(array[1], array[6]);
    array[6] = std::max(array[1], array[6]);
    array[1] = tmp;
    tmp = std::min(array[2], array[4]);
    array[4] = std::max(array[2], array[4]);
    array[2] = tmp;
    array[1] = std::max(array[0], array[1]);
    tmp = std::min(array[3], array[5]);
    array[5] = std::max(array[3], array[5]);
    array[3] = tmp;
    tmp = std::min(array[2], array[6]);
    array[6] = std::max(array[2], array[6]);
    array[3] = std::max(tmp, array[3]);
    array[3] = std::min(array[3], array[6]);
    tmp = std::min(array[4], array[5]);
    array[4] = std::max(array[1], tmp);
    tmp = std::min(array[1], tmp);
    array[3] = std::max(tmp, array[3]);
    return std::min(array[3], array[4]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 7> array)
{
    vfloat tmp = vminf(array[0], array[5]);
    array[5] = vmaxf(array[0], array[5]);
    array[0] = tmp;
    tmp = vminf(array[0], array[3]);
    array[3] = vmaxf(array[0], array[3]);
    array[0] = tmp;
    tmp = vminf(array[1], array[6]);
    array[6] = vmaxf(array[1], array[6]);
    array[1] = tmp;
    tmp = vminf(array[2], array[4]);
    array[4] = vmaxf(array[2], array[4]);
    array[2] = tmp;
    array[1] = vmaxf(array[0], array[1]);
    tmp = vminf(array[3], array[5]);
    array[5] = vmaxf(array[3], array[5]);
    array[3] = tmp;
    tmp = vminf(array[2], array[6]);
    array[6] = vmaxf(array[2], array[6]);
    array[3] = vmaxf(tmp, array[3]);
    array[3] = vminf(array[3], array[6]);
    tmp = vminf(array[4], array[5]);
    array[4] = vmaxf(array[1], tmp);
    tmp = vminf(array[1], tmp);
    array[3] = vmaxf(tmp, array[3]);
    return vminf(array[3], array[4]);
}
#endif

template<typename T>
inline T median(std::array<T, 9> array)
{
    T tmp = std::min(array[1], array[2]);
    array[2] = std::max(array[1], array[2]);
    array[1] = tmp;
    tmp = std::min(array[4], array[5]);
    array[5] = std::max(array[4], array[5]);
    array[4] = tmp;
    tmp = std::min(array[7], array[8]);
    array[8] = std::max(array[7], array[8]);
    array[7] = tmp;
    tmp = std::min(array[0], array[1]);
    array[1] = std::max(array[0], array[1]);
    array[0] = tmp;
    tmp = std::min(array[3], array[4]);
    array[4] = std::max(array[3], array[4]);
    array[3] = tmp;
    tmp = std::min(array[6], array[7]);
    array[7] = std::max(array[6], array[7]);
    array[6] = tmp;
    tmp = std::min(array[1], array[2]);
    array[2] = std::max(array[1], array[2]);
    array[1] = tmp;
    tmp = std::min(array[4], array[5]);
    array[5] = std::max(array[4], array[5]);
    array[4] = tmp;
    tmp = std::min(array[7], array[8]);
    array[8] = std::max(array[7], array[8]);
    array[3] = std::max(array[0], array[3]);
    array[5] = std::min(array[5], array[8]);
    array[7] = std::max(array[4], tmp);
    tmp = std::min(array[4], tmp);
    array[6] = std::max(array[3], array[6]);
    array[4] = std::max(array[1], tmp);
    array[2] = std::min(array[2], array[5]);
    array[4] = std::min(array[4], array[7]);
    tmp = std::min(array[4], array[2]);
    array[2] = std::max(array[4], array[2]);
    array[4] = std::max(array[6], tmp);
    return std::min(array[4], array[2]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 9> array)
{
    vfloat tmp = vminf(array[1], array[2]);
    array[2] = vmaxf(array[1], array[2]);
    array[1] = tmp;
    tmp = vminf(array[4], array[5]);
    array[5] = vmaxf(array[4], array[5]);
    array[4] = tmp;
    tmp = vminf(array[7], array[8]);
    array[8] = vmaxf(array[7], array[8]);
    array[7] = tmp;
    tmp = vminf(array[0], array[1]);
    array[1] = vmaxf(array[0], array[1]);
    array[0] = tmp;
    tmp = vminf(array[3], array[4]);
    array[4] = vmaxf(array[3], array[4]);
    array[3] = tmp;
    tmp = vminf(array[6], array[7]);
    array[7] = vmaxf(array[6], array[7]);
    array[6] = tmp;
    tmp = vminf(array[1], array[2]);
    array[2] = vmaxf(array[1], array[2]);
    array[1] = tmp;
    tmp = vminf(array[4], array[5]);
    array[5] = vmaxf(array[4], array[5]);
    array[4] = tmp;
    tmp = vminf(array[7], array[8]);
    array[8] = vmaxf(array[7], array[8]);
    array[3] = vmaxf(array[0], array[3]);
    array[5] = vminf(array[5], array[8]);
    array[7] = vmaxf(array[4], tmp);
    tmp = vminf(array[4], tmp);
    array[6] = vmaxf(array[3], array[6]);
    array[4] = vmaxf(array[1], tmp);
    array[2] = vminf(array[2], array[5]);
    array[4] = vminf(array[4], array[7]);
    tmp = vminf(array[4], array[2]);
    array[2] = vmaxf(array[4], array[2]);
    array[4] = vmaxf(array[6], tmp);
    return vminf(array[4], array[2]);
}
#endif

template<typename T>
inline T median(std::array<T, 13> array)
{
    T tmp = std::min(array[1], array[7]);
    array[7] = std::max(array[1], array[7]);
    array[1] = tmp;
    tmp = std::min(array[9], array[11]);
    array[11] = std::max(array[9], array[11]);
    array[9] = tmp;
    tmp = std::min(array[3], array[4]);
    array[4] = std::max(array[3], array[4]);
    array[3] = tmp;
    tmp = std::min(array[5], array[8]);
    array[8] = std::max(array[5], array[8]);
    array[5] = tmp;
    tmp = std::min(array[0], array[12]);
    array[12] = std::max(array[0], array[12]);
    array[0] = tmp;
    tmp = std::min(array[2], array[6]);
    array[6] = std::max(array[2], array[6]);
    array[2] = tmp;
    tmp = std::min(array[0], array[1]);
    array[1] = std::max(array[0], array[1]);
    array[0] = tmp;
    tmp = std::min(array[2], array[3]);
    array[3] = std::max(array[2], array[3]);
    array[2] = tmp;
    tmp = std::min(array[4], array[6]);
    array[6] = std::max(array[4], array[6]);
    array[4] = tmp;
    tmp = std::min(array[8], array[11]);
    array[11] = std::max(array[8], array[11]);
    array[8] = tmp;
    tmp = std::min(array[7], array[12]);
    array[12] = std::max(array[7], array[12]);
    array[7] = tmp;
    tmp = std::min(array[5], array[9]);
    array[9] = std::max(array[5], array[9]);
    array[5] = tmp;
    tmp = std::min(array[0], array[2]);
    array[2] = std::max(array[0], array[2]);
    array[0] = tmp;
    tmp = std::min(array[3], array[7]);
    array[7] = std::max(array[3], array[7]);
    array[3] = tmp;
    tmp = std::min(array[10], array[11]);
    array[11] = std::max(array[10], array[11]);
    array[10] = tmp;
    tmp = std::min(array[1], array[4]);
    array[4] = std::max(array[1], array[4]);
    array[1] = tmp;
    tmp = std::min(array[6], array[12]);
    array[12] = std::max(array[6], array[12]);
    array[6] = tmp;
    tmp = std::min(array[7], array[8]);
    array[8] = std::max(array[7], array[8]);
    array[7] = tmp;
    array[11] = std::min(array[11], array[12]);
    tmp = std::min(array[4], array[9]);
    array[9] = std::max(array[4], array[9]);
    array[4] = tmp;
    tmp = std::min(array[6], array[10]);
    array[10] = std::max(array[6], array[10]);
    array[6] = tmp;
    tmp = std::min(array[3], array[4]);
    array[4] = std::max(array[3], array[4]);
    array[3] = tmp;
    tmp = std::min(array[5], array[6]);
    array[6] = std::max(array[5], array[6]);
    array[5] = tmp;
    array[8] = std::min(array[8], array[9]);
    array[10] = std::min(array[10], array[11]);
    tmp = std::min(array[1], array[7]);
    array[7] = std::max(array[1], array[7]);
    array[1] = tmp;
    tmp = std::min(array[2], array[6]);
    array[6] = std::max(array[2], array[6]);
    array[2] = tmp;
    array[3] = std::max(array[1], array[3]);
    tmp = std::min(array[4], array[7]);
    array[7] = std::max(array[4], array[7]);
    array[4] = tmp;
    array[8] = std::min(array[8], array[10]);
    array[5] = std::max(array[0], array[5]);
    array[5] = std::max(array[2], array[5]);
    tmp = std::min(array[6], array[8]);
    array[8] = std::max(array[6], array[8]);
    array[5] = std::max(array[3], array[5]);
    array[7] = std::min(array[7], array[8]);
    array[6] = std::max(array[4], tmp);
    tmp = std::min(array[4], tmp);
    array[5] = std::max(tmp, array[5]);
    array[6] = std::min(array[6], array[7]);
    return std::max(array[5], array[6]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 13> array)
{
    vfloat tmp = vminf(array[1], array[7]);
    array[7] = vmaxf(array[1], array[7]);
    array[1] = tmp;
    tmp = vminf(array[9], array[11]);
    array[11] = vmaxf(array[9], array[11]);
    array[9] = tmp;
    tmp = vminf(array[3], array[4]);
    array[4] = vmaxf(array[3], array[4]);
    array[3] = tmp;
    tmp = vminf(array[5], array[8]);
    array[8] = vmaxf(array[5], array[8]);
    array[5] = tmp;
    tmp = vminf(array[0], array[12]);
    array[12] = vmaxf(array[0], array[12]);
    array[0] = tmp;
    tmp = vminf(array[2], array[6]);
    array[6] = vmaxf(array[2], array[6]);
    array[2] = tmp;
    tmp = vminf(array[0], array[1]);
    array[1] = vmaxf(array[0], array[1]);
    array[0] = tmp;
    tmp = vminf(array[2], array[3]);
    array[3] = vmaxf(array[2], array[3]);
    array[2] = tmp;
    tmp = vminf(array[4], array[6]);
    array[6] = vmaxf(array[4], array[6]);
    array[4] = tmp;
    tmp = vminf(array[8], array[11]);
    array[11] = vmaxf(array[8], array[11]);
    array[8] = tmp;
    tmp = vminf(array[7], array[12]);
    array[12] = vmaxf(array[7], array[12]);
    array[7] = tmp;
    tmp = vminf(array[5], array[9]);
    array[9] = vmaxf(array[5], array[9]);
    array[5] = tmp;
    tmp = vminf(array[0], array[2]);
    array[2] = vmaxf(array[0], array[2]);
    array[0] = tmp;
    tmp = vminf(array[3], array[7]);
    array[7] = vmaxf(array[3], array[7]);
    array[3] = tmp;
    tmp = vminf(array[10], array[11]);
    array[11] = vmaxf(array[10], array[11]);
    array[10] = tmp;
    tmp = vminf(array[1], array[4]);
    array[4] = vmaxf(array[1], array[4]);
    array[1] = tmp;
    tmp = vminf(array[6], array[12]);
    array[12] = vmaxf(array[6], array[12]);
    array[6] = tmp;
    tmp = vminf(array[7], array[8]);
    array[8] = vmaxf(array[7], array[8]);
    array[7] = tmp;
    array[11] = vminf(array[11], array[12]);
    tmp = vminf(array[4], array[9]);
    array[9] = vmaxf(array[4], array[9]);
    array[4] = tmp;
    tmp = vminf(array[6], array[10]);
    array[10] = vmaxf(array[6], array[10]);
    array[6] = tmp;
    tmp = vminf(array[3], array[4]);
    array[4] = vmaxf(array[3], array[4]);
    array[3] = tmp;
    tmp = vminf(array[5], array[6]);
    array[6] = vmaxf(array[5], array[6]);
    array[5] = tmp;
    array[8] = vminf(array[8], array[9]);
    array[10] = vminf(array[10], array[11]);
    tmp = vminf(array[1], array[7]);
    array[7] = vmaxf(array[1], array[7]);
    array[1] = tmp;
    tmp = vminf(array[2], array[6]);
    array[6] = vmaxf(array[2], array[6]);
    array[2] = tmp;
    array[3] = vmaxf(array[1], array[3]);
    tmp = vminf(array[4], array[7]);
    array[7] = vmaxf(array[4], array[7]);
    array[4] = tmp;
    array[8] = vminf(array[8], array[10]);
    array[5] = vmaxf(array[0], array[5]);
    array[5] = vmaxf(array[2], array[5]);
    tmp = vminf(array[6], array[8]);
    array[8] = vmaxf(array[6], array[8]);
    array[5] = vmaxf(array[3], array[5]);
    array[7] = vminf(array[7], array[8]);
    array[6] = vmaxf(array[4], tmp);
    tmp = vminf(array[4], tmp);
    array[5] = vmaxf(tmp, array[5]);
    array[6] = vminf(array[6], array[7]);
    return vmaxf(array[5], array[6]);
}
#endif

template<typename T>
inline T median(std::array<T, 25> array)
{
    T tmp = std::min(array[0], array[1]);
    array[1] = std::max(array[0], array[1]);
    array[0] = tmp;
    tmp = std::min(array[3], array[4]);
    array[4] = std::max(array[3], array[4]);
    array[3] = tmp;
    tmp = std::min(array[2], array[4]);
    array[4] = std::max(array[2], array[4]);
    array[2] = std::min(tmp, array[3]);
    array[3] = std::max(tmp, array[3]);
    tmp = std::min(array[6], array[7]);
    array[7] = std::max(array[6], array[7]);
    array[6] = tmp;
    tmp = std::min(array[5], array[7]);
    array[7] = std::max(array[5], array[7]);
    array[5] = std::min(tmp, array[6]);
    array[6] = std::max(tmp, array[6]);
    tmp = std::min(array[9], array[10]);
    array[10] = std::max(array[9], array[10]);
    array[9] = tmp;
    tmp = std::min(array[8], array[10]);
    array[10] = std::max(array[8], array[10]);
    array[8] = std::min(tmp, array[9]);
    array[9] = std::max(tmp, array[9]);
    tmp = std::min(array[12], array[13]);
    array[13] = std::max(array[12], array[13]);
    array[12] = tmp;
    tmp = std::min(array[11], array[13]);
    array[13] = std::max(array[11], array[13]);
    array[11] = std::min(tmp, array[12]);
    array[12] = std::max(tmp, array[12]);
    tmp = std::min(array[15], array[16]);
    array[16] = std::max(array[15], array[16]);
    array[15] = tmp;
    tmp = std::min(array[14], array[16]);
    array[16] = std::max(array[14], array[16]);
    array[14] = std::min(tmp, array[15]);
    array[15] = std::max(tmp, array[15]);
    tmp = std::min(array[18], array[19]);
    array[19] = std::max(array[18], array[19]);
    array[18] = tmp;
    tmp = std::min(array[17], array[19]);
    array[19] = std::max(array[17], array[19]);
    array[17] = std::min(tmp, array[18]);
    array[18] = std::max(tmp, array[18]);
    tmp = std::min(array[21], array[22]);
    array[22] = std::max(array[21], array[22]);
    array[21] = tmp;
    tmp = std::min(array[20], array[22]);
    array[22] = std::max(array[20], array[22]);
    array[20] = std::min(tmp, array[21]);
    array[21] = std::max(tmp, array[21]);
    tmp = std::min(array[23], array[24]);
    array[24] = std::max(array[23], array[24]);
    array[23] = tmp;
    tmp = std::min(array[2], array[5]);
    array[5] = std::max(array[2], array[5]);
    array[2] = tmp;
    tmp = std::min(array[3], array[6]);
    array[6] = std::max(array[3], array[6]);
    array[3] = tmp;
    tmp = std::min(array[0], array[6]);
    array[6] = std::max(array[0], array[6]);
    array[0] = std::min(tmp, array[3]);
    array[3] = std::max(tmp, array[3]);
    tmp = std::min(array[4], array[7]);
    array[7] = std::max(array[4], array[7]);
    array[4] = tmp;
    tmp = std::min(array[1], array[7]);
    array[7] = std::max(array[1], array[7]);
    array[1] = std::min(tmp, array[4]);
    array[4] = std::max(tmp, array[4]);
    tmp = std::min(array[11], array[14]);
    array[14] = std::max(array[11], array[14]);
    array[11] = tmp;
    tmp = std::min(array[8], array[14]);
    array[14] = std::max(array[8], array[14]);
    array[8] = std::min(tmp, array[11]);
    array[11] = std::max(tmp, array[11]);
    tmp = std::min(array[12], array[15]);
    array[15] = std::max(array[12], array[15]);
    array[12] = tmp;
    tmp = std::min(array[9], array[15]);
    array[15] = std::max(array[9], array[15]);
    array[9] = std::min(tmp, array[12]);
    array[12] = std::max(tmp, array[12]);
    tmp = std::min(array[13], array[16]);
    array[16] = std::max(array[13], array[16]);
    array[13] = tmp;
    tmp = std::min(array[10], array[16]);
    array[16] = std::max(array[10], array[16]);
    array[10] = std::min(tmp, array[13]);
    array[13] = std::max(tmp, array[13]);
    tmp = std::min(array[20], array[23]);
    array[23] = std::max(array[20], array[23]);
    array[20] = tmp;
    tmp = std::min(array[17], array[23]);
    array[23] = std::max(array[17], array[23]);
    array[17] = std::min(tmp, array[20]);
    array[20] = std::max(tmp, array[20]);
    tmp = std::min(array[21], array[24]);
    array[24] = std::max(array[21], array[24]);
    array[21] = tmp;
    tmp = std::min(array[18], array[24]);
    array[24] = std::max(array[18], array[24]);
    array[18] = std::min(tmp, array[21]);
    array[21] = std::max(tmp, array[21]);
    tmp = std::min(array[19], array[22]);
    array[22] = std::max(array[19], array[22]);
    array[19] = tmp;
    array[17] = std::max(array[8], array[17]);
    tmp = std::min(array[9], array[18]);
    array[18] = std::max(array[9], array[18]);
    array[9] = tmp;
    tmp = std::min(array[0], array[18]);
    array[18] = std::max(array[0], array[18]);
    array[9] = std::max(tmp, array[9]);
    tmp = std::min(array[10], array[19]);
    array[19] = std::max(array[10], array[19]);
    array[10] = tmp;
    tmp = std::min(array[1], array[19]);
    array[19] = std::max(array[1], array[19]);
    array[1] = std::min(tmp, array[10]);
    array[10] = std::max(tmp, array[10]);
    tmp = std::min(array[11], array[20]);
    array[20] = std::max(array[11], array[20]);
    array[11] = tmp;
    tmp = std::min(array[2], array[20]);
    array[20] = std::max(array[2], array[20]);
    array[11] = std::max(tmp, array[11]);
    tmp = std::min(array[12], array[21]);
    array[21] = std::max(array[12], array[21]);
    array[12] = tmp;
    tmp = std::min(array[3], array[21]);
    array[21] = std::max(array[3], array[21]);
    array[3] = std::min(tmp, array[12]);
    array[12] = std::max(tmp, array[12]);
    tmp = std::min(array[13], array[22]);
    array[22] = std::max(array[13], array[22]);
    array[4] = std::min(array[4], array[22]);
    array[13] = std::max(array[4], tmp);
    tmp = std::min(array[4], tmp);
    array[4] = tmp;
    tmp = std::min(array[14], array[23]);
    array[23] = std::max(array[14], array[23]);
    array[14] = tmp;
    tmp = std::min(array[5], array[23]);
    array[23] = std::max(array[5], array[23]);
    array[5] = std::min(tmp, array[14]);
    array[14] = std::max(tmp, array[14]);
    tmp = std::min(array[15], array[24]);
    array[24] = std::max(array[15], array[24]);
    array[15] = tmp;
    array[6] = std::min(array[6], array[24]);
    tmp = std::min(array[6], array[15]);
    array[15] = std::max(array[6], array[15]);
    array[6] = tmp;
    tmp = std::min(array[7], array[16]);
    array[7] = std::min(tmp, array[19]);
    tmp = std::min(array[13], array[21]);
    array[15] = std::min(array[15], array[23]);
    tmp = std::min(array[7], tmp);
    array[7] = std::min(tmp, array[15]);
    array[9] = std::max(array[1], array[9]);
    array[11] = std::max(array[3], array[11]);
    array[17] = std::max(array[5], array[17]);
    array[17] = std::max(array[11], array[17]);
    array[17] = std::max(array[9], array[17]);
    tmp = std::min(array[4], array[10]);
    array[10] = std::max(array[4], array[10]);
    array[4] = tmp;
    tmp = std::min(array[6], array[12]);
    array[12] = std::max(array[6], array[12]);
    array[6] = tmp;
    tmp = std::min(array[7], array[14]);
    array[14] = std::max(array[7], array[14]);
    array[7] = tmp;
    tmp = std::min(array[4], array[6]);
    array[6] = std::max(array[4], array[6]);
    array[7] = std::max(tmp, array[7]);
    tmp = std::min(array[12], array[14]);
    array[14] = std::max(array[12], array[14]);
    array[12] = tmp;
    array[10] = std::min(array[10], array[14]);
    tmp = std::min(array[6], array[7]);
    array[7] = std::max(array[6], array[7]);
    array[6] = tmp;
    tmp = std::min(array[10], array[12]);
    array[12] = std::max(array[10], array[12]);
    array[10] = std::max(array[6], tmp);
    tmp = std::min(array[6], tmp);
    array[17] = std::max(tmp, array[17]);
    tmp = std::min(array[12], array[17]);
    array[17] = std::max(array[12], array[17]);
    array[12] = tmp;
    array[7] = std::min(array[7], array[17]);
    tmp = std::min(array[7], array[10]);
    array[10] = std::max(array[7], array[10]);
    array[7] = tmp;
    tmp = std::min(array[12], array[18]);
    array[18] = std::max(array[12], array[18]);
    array[12] = std::max(array[7], tmp);
    array[10] = std::min(array[10], array[18]);
    tmp = std::min(array[12], array[20]);
    array[20] = std::max(array[12], array[20]);
    array[12] = tmp;
    tmp = std::min(array[10], array[20]);
    return std::max(tmp, array[12]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 25> array)
{
    vfloat tmp = vminf(array[0], array[1]);
    array[1] = vmaxf(array[0], array[1]);
    array[0] = tmp;
    tmp = vminf(array[3], array[4]);
    array[4] = vmaxf(array[3], array[4]);
    array[3] = tmp;
    tmp = vminf(array[2], array[4]);
    array[4] = vmaxf(array[2], array[4]);
    array[2] = vminf(tmp, array[3]);
    array[3] = vmaxf(tmp, array[3]);
    tmp = vminf(array[6], array[7]);
    array[7] = vmaxf(array[6], array[7]);
    array[6] = tmp;
    tmp = vminf(array[5], array[7]);
    array[7] = vmaxf(array[5], array[7]);
    array[5] = vminf(tmp, array[6]);
    array[6] = vmaxf(tmp, array[6]);
    tmp = vminf(array[9], array[10]);
    array[10] = vmaxf(array[9], array[10]);
    array[9] = tmp;
    tmp = vminf(array[8], array[10]);
    array[10] = vmaxf(array[8], array[10]);
    array[8] = vminf(tmp, array[9]);
    array[9] = vmaxf(tmp, array[9]);
    tmp = vminf(array[12], array[13]);
    array[13] = vmaxf(array[12], array[13]);
    array[12] = tmp;
    tmp = vminf(array[11], array[13]);
    array[13] = vmaxf(array[11], array[13]);
    array[11] = vminf(tmp, array[12]);
    array[12] = vmaxf(tmp, array[12]);
    tmp = vminf(array[15], array[16]);
    array[16] = vmaxf(array[15], array[16]);
    array[15] = tmp;
    tmp = vminf(array[14], array[16]);
    array[16] = vmaxf(array[14], array[16]);
    array[14] = vminf(tmp, array[15]);
    array[15] = vmaxf(tmp, array[15]);
    tmp = vminf(array[18], array[19]);
    array[19] = vmaxf(array[18], array[19]);
    array[18] = tmp;
    tmp = vminf(array[17], array[19]);
    array[19] = vmaxf(array[17], array[19]);
    array[17] = vminf(tmp, array[18]);
    array[18] = vmaxf(tmp, array[18]);
    tmp = vminf(array[21], array[22]);
    array[22] = vmaxf(array[21], array[22]);
    array[21] = tmp;
    tmp = vminf(array[20], array[22]);
    array[22] = vmaxf(array[20], array[22]);
    array[20] = vminf(tmp, array[21]);
    array[21] = vmaxf(tmp, array[21]);
    tmp = vminf(array[23], array[24]);
    array[24] = vmaxf(array[23], array[24]);
    array[23] = tmp;
    tmp = vminf(array[2], array[5]);
    array[5] = vmaxf(array[2], array[5]);
    array[2] = tmp;
    tmp = vminf(array[3], array[6]);
    array[6] = vmaxf(array[3], array[6]);
    array[3] = tmp;
    tmp = vminf(array[0], array[6]);
    array[6] = vmaxf(array[0], array[6]);
    array[0] = vminf(tmp, array[3]);
    array[3] = vmaxf(tmp, array[3]);
    tmp = vminf(array[4], array[7]);
    array[7] = vmaxf(array[4], array[7]);
    array[4] = tmp;
    tmp = vminf(array[1], array[7]);
    array[7] = vmaxf(array[1], array[7]);
    array[1] = vminf(tmp, array[4]);
    array[4] = vmaxf(tmp, array[4]);
    tmp = vminf(array[11], array[14]);
    array[14] = vmaxf(array[11], array[14]);
    array[11] = tmp;
    tmp = vminf(array[8], array[14]);
    array[14] = vmaxf(array[8], array[14]);
    array[8] = vminf(tmp, array[11]);
    array[11] = vmaxf(tmp, array[11]);
    tmp = vminf(array[12], array[15]);
    array[15] = vmaxf(array[12], array[15]);
    array[12] = tmp;
    tmp = vminf(array[9], array[15]);
    array[15] = vmaxf(array[9], array[15]);
    array[9] = vminf(tmp, array[12]);
    array[12] = vmaxf(tmp, array[12]);
    tmp = vminf(array[13], array[16]);
    array[16] = vmaxf(array[13], array[16]);
    array[13] = tmp;
    tmp = vminf(array[10], array[16]);
    array[16] = vmaxf(array[10], array[16]);
    array[10] = vminf(tmp, array[13]);
    array[13] = vmaxf(tmp, array[13]);
    tmp = vminf(array[20], array[23]);
    array[23] = vmaxf(array[20], array[23]);
    array[20] = tmp;
    tmp = vminf(array[17], array[23]);
    array[23] = vmaxf(array[17], array[23]);
    array[17] = vminf(tmp, array[20]);
    array[20] = vmaxf(tmp, array[20]);
    tmp = vminf(array[21], array[24]);
    array[24] = vmaxf(array[21], array[24]);
    array[21] = tmp;
    tmp = vminf(array[18], array[24]);
    array[24] = vmaxf(array[18], array[24]);
    array[18] = vminf(tmp, array[21]);
    array[21] = vmaxf(tmp, array[21]);
    tmp = vminf(array[19], array[22]);
    array[22] = vmaxf(array[19], array[22]);
    array[19] = tmp;
    array[17] = vmaxf(array[8], array[17]);
    tmp = vminf(array[9], array[18]);
    array[18] = vmaxf(array[9], array[18]);
    array[9] = tmp;
    tmp = vminf(array[0], array[18]);
    array[18] = vmaxf(array[0], array[18]);
    array[9] = vmaxf(tmp, array[9]);
    tmp = vminf(array[10], array[19]);
    array[19] = vmaxf(array[10], array[19]);
    array[10] = tmp;
    tmp = vminf(array[1], array[19]);
    array[19] = vmaxf(array[1], array[19]);
    array[1] = vminf(tmp, array[10]);
    array[10] = vmaxf(tmp, array[10]);
    tmp = vminf(array[11], array[20]);
    array[20] = vmaxf(array[11], array[20]);
    array[11] = tmp;
    tmp = vminf(array[2], array[20]);
    array[20] = vmaxf(array[2], array[20]);
    array[11] = vmaxf(tmp, array[11]);
    tmp = vminf(array[12], array[21]);
    array[21] = vmaxf(array[12], array[21]);
    array[12] = tmp;
    tmp = vminf(array[3], array[21]);
    array[21] = vmaxf(array[3], array[21]);
    array[3] = vminf(tmp, array[12]);
    array[12] = vmaxf(tmp, array[12]);
    tmp = vminf(array[13], array[22]);
    array[22] = vmaxf(array[13], array[22]);
    array[4] = vminf(array[4], array[22]);
    array[13] = vmaxf(array[4], tmp);
    tmp = vminf(array[4], tmp);
    array[4] = tmp;
    tmp = vminf(array[14], array[23]);
    array[23] = vmaxf(array[14], array[23]);
    array[14] = tmp;
    tmp = vminf(array[5], array[23]);
    array[23] = vmaxf(array[5], array[23]);
    array[5] = vminf(tmp, array[14]);
    array[14] = vmaxf(tmp, array[14]);
    tmp = vminf(array[15], array[24]);
    array[24] = vmaxf(array[15], array[24]);
    array[15] = tmp;
    array[6] = vminf(array[6], array[24]);
    tmp = vminf(array[6], array[15]);
    array[15] = vmaxf(array[6], array[15]);
    array[6] = tmp;
    tmp = vminf(array[7], array[16]);
    array[7] = vminf(tmp, array[19]);
    tmp = vminf(array[13], array[21]);
    array[15] = vminf(array[15], array[23]);
    tmp = vminf(array[7], tmp);
    array[7] = vminf(tmp, array[15]);
    array[9] = vmaxf(array[1], array[9]);
    array[11] = vmaxf(array[3], array[11]);
    array[17] = vmaxf(array[5], array[17]);
    array[17] = vmaxf(array[11], array[17]);
    array[17] = vmaxf(array[9], array[17]);
    tmp = vminf(array[4], array[10]);
    array[10] = vmaxf(array[4], array[10]);
    array[4] = tmp;
    tmp = vminf(array[6], array[12]);
    array[12] = vmaxf(array[6], array[12]);
    array[6] = tmp;
    tmp = vminf(array[7], array[14]);
    array[14] = vmaxf(array[7], array[14]);
    array[7] = tmp;
    tmp = vminf(array[4], array[6]);
    array[6] = vmaxf(array[4], array[6]);
    array[7] = vmaxf(tmp, array[7]);
    tmp = vminf(array[12], array[14]);
    array[14] = vmaxf(array[12], array[14]);
    array[12] = tmp;
    array[10] = vminf(array[10], array[14]);
    tmp = vminf(array[6], array[7]);
    array[7] = vmaxf(array[6], array[7]);
    array[6] = tmp;
    tmp = vminf(array[10], array[12]);
    array[12] = vmaxf(array[10], array[12]);
    array[10] = vmaxf(array[6], tmp);
    tmp = vminf(array[6], tmp);
    array[17] = vmaxf(tmp, array[17]);
    tmp = vminf(array[12], array[17]);
    array[17] = vmaxf(array[12], array[17]);
    array[12] = tmp;
    array[7] = vminf(array[7], array[17]);
    tmp = vminf(array[7], array[10]);
    array[10] = vmaxf(array[7], array[10]);
    array[7] = tmp;
    tmp = vminf(array[12], array[18]);
    array[18] = vmaxf(array[12], array[18]);
    array[12] = vmaxf(array[7], tmp);
    array[10] = vminf(array[10], array[18]);
    tmp = vminf(array[12], array[20]);
    array[20] = vmaxf(array[12], array[20]);
    array[12] = tmp;
    tmp = vminf(array[10], array[20]);
    return vmaxf(tmp, array[12]);
}
#endif

template<typename T>
inline T median(std::array<T, 49> array)
{
    T tmp = std::min(array[0], array[32]);
    array[32] = std::max(array[0], array[32]);
    array[0] = tmp;
    tmp = std::min(array[1], array[33]);
    array[33] = std::max(array[1], array[33]);
    array[1] = tmp;
    tmp = std::min(array[2], array[34]);
    array[34] = std::max(array[2], array[34]);
    array[2] = tmp;
    tmp = std::min(array[3], array[35]);
    array[35] = std::max(array[3], array[35]);
    array[3] = tmp;
    tmp = std::min(array[4], array[36]);
    array[36] = std::max(array[4], array[36]);
    array[4] = tmp;
    tmp = std::min(array[5], array[37]);
    array[37] = std::max(array[5], array[37]);
    array[5] = tmp;
    tmp = std::min(array[6], array[38]);
    array[38] = std::max(array[6], array[38]);
    array[6] = tmp;
    tmp = std::min(array[7], array[39]);
    array[39] = std::max(array[7], array[39]);
    array[7] = tmp;
    tmp = std::min(array[8], array[40]);
    array[40] = std::max(array[8], array[40]);
    array[8] = tmp;
    tmp = std::min(array[9], array[41]);
    array[41] = std::max(array[9], array[41]);
    array[9] = tmp;
    tmp = std::min(array[10], array[42]);
    array[42] = std::max(array[10], array[42]);
    array[10] = tmp;
    tmp = std::min(array[11], array[43]);
    array[43] = std::max(array[11], array[43]);
    array[11] = tmp;
    tmp = std::min(array[12], array[44]);
    array[44] = std::max(array[12], array[44]);
    array[12] = tmp;
    tmp = std::min(array[13], array[45]);
    array[45] = std::max(array[13], array[45]);
    array[13] = tmp;
    tmp = std::min(array[14], array[46]);
    array[46] = std::max(array[14], array[46]);
    array[14] = tmp;
    tmp = std::min(array[15], array[47]);
    array[47] = std::max(array[15], array[47]);
    array[15] = tmp;
    tmp = std::min(array[16], array[48]);
    array[48] = std::max(array[16], array[48]);
    array[16] = tmp;
    tmp = std::min(array[0], array[16]);
    array[16] = std::max(array[0], array[16]);
    array[0] = tmp;
    tmp = std::min(array[1], array[17]);
    array[17] = std::max(array[1], array[17]);
    array[1] = tmp;
    tmp = std::min(array[2], array[18]);
    array[18] = std::max(array[2], array[18]);
    array[2] = tmp;
    tmp = std::min(array[3], array[19]);
    array[19] = std::max(array[3], array[19]);
    array[3] = tmp;
    tmp = std::min(array[4], array[20]);
    array[20] = std::max(array[4], array[20]);
    array[4] = tmp;
    tmp = std::min(array[5], array[21]);
    array[21] = std::max(array[5], array[21]);
    array[5] = tmp;
    tmp = std::min(array[6], array[22]);
    array[22] = std::max(array[6], array[22]);
    array[6] = tmp;
    tmp = std::min(array[7], array[23]);
    array[23] = std::max(array[7], array[23]);
    array[7] = tmp;
    tmp = std::min(array[8], array[24]);
    array[24] = std::max(array[8], array[24]);
    array[8] = tmp;
    tmp = std::min(array[9], array[25]);
    array[25] = std::max(array[9], array[25]);
    array[9] = tmp;
    tmp = std::min(array[10], array[26]);
    array[26] = std::max(array[10], array[26]);
    array[10] = tmp;
    tmp = std::min(array[11], array[27]);
    array[27] = std::max(array[11], array[27]);
    array[11] = tmp;
    tmp = std::min(array[12], array[28]);
    array[28] = std::max(array[12], array[28]);
    array[12] = tmp;
    tmp = std::min(array[13], array[29]);
    array[29] = std::max(array[13], array[29]);
    array[13] = tmp;
    tmp = std::min(array[14], array[30]);
    array[30] = std::max(array[14], array[30]);
    array[14] = tmp;
    tmp = std::min(array[15], array[31]);
    array[31] = std::max(array[15], array[31]);
    array[15] = tmp;
    tmp = std::min(array[32], array[48]);
    array[48] = std::max(array[32], array[48]);
    array[32] = tmp;
    tmp = std::min(array[16], array[32]);
    array[32] = std::max(array[16], array[32]);
    array[16] = tmp;
    tmp = std::min(array[17], array[33]);
    array[33] = std::max(array[17], array[33]);
    array[17] = tmp;
    tmp = std::min(array[18], array[34]);
    array[34] = std::max(array[18], array[34]);
    array[18] = tmp;
    tmp = std::min(array[19], array[35]);
    array[35] = std::max(array[19], array[35]);
    array[19] = tmp;
    tmp = std::min(array[20], array[36]);
    array[36] = std::max(array[20], array[36]);
    array[20] = tmp;
    tmp = std::min(array[21], array[37]);
    array[37] = std::max(array[21], array[37]);
    array[21] = tmp;
    tmp = std::min(array[22], array[38]);
    array[38] = std::max(array[22], array[38]);
    array[22] = tmp;
    tmp = std::min(array[23], array[39]);
    array[39] = std::max(array[23], array[39]);
    array[23] = tmp;
    tmp = std::min(array[24], array[40]);
    array[40] = std::max(array[24], array[40]);
    array[24] = tmp;
    tmp = std::min(array[25], array[41]);
    array[41] = std::max(array[25], array[41]);
    array[25] = tmp;
    tmp = std::min(array[26], array[42]);
    array[42] = std::max(array[26], array[42]);
    array[26] = tmp;
    tmp = std::min(array[27], array[43]);
    array[43] = std::max(array[27], array[43]);
    array[27] = tmp;
    tmp = std::min(array[28], array[44]);
    array[44] = std::max(array[28], array[44]);
    array[28] = tmp;
    tmp = std::min(array[29], array[45]);
    array[45] = std::max(array[29], array[45]);
    array[29] = tmp;
    tmp = std::min(array[30], array[46]);
    array[46] = std::max(array[30], array[46]);
    array[30] = tmp;
    tmp = std::min(array[31], array[47]);
    array[47] = std::max(array[31], array[47]);
    array[31] = tmp;
    tmp = std::min(array[0], array[8]);
    array[8] = std::max(array[0], array[8]);
    array[0] = tmp;
    tmp = std::min(array[1], array[9]);
    array[9] = std::max(array[1], array[9]);
    array[1] = tmp;
    tmp = std::min(array[2], array[10]);
    array[10] = std::max(array[2], array[10]);
    array[2] = tmp;
    tmp = std::min(array[3], array[11]);
    array[11] = std::max(array[3], array[11]);
    array[3] = tmp;
    tmp = std::min(array[4], array[12]);
    array[12] = std::max(array[4], array[12]);
    array[4] = tmp;
    tmp = std::min(array[5], array[13]);
    array[13] = std::max(array[5], array[13]);
    array[5] = tmp;
    tmp = std::min(array[6], array[14]);
    array[14] = std::max(array[6], array[14]);
    array[6] = tmp;
    tmp = std::min(array[7], array[15]);
    array[15] = std::max(array[7], array[15]);
    array[7] = tmp;
    tmp = std::min(array[16], array[24]);
    array[24] = std::max(array[16], array[24]);
    array[16] = tmp;
    tmp = std::min(array[17], array[25]);
    array[25] = std::max(array[17], array[25]);
    array[17] = tmp;
    tmp = std::min(array[18], array[26]);
    array[26] = std::max(array[18], array[26]);
    array[18] = tmp;
    tmp = std::min(array[19], array[27]);
    array[27] = std::max(array[19], array[27]);
    array[19] = tmp;
    tmp = std::min(array[20], array[28]);
    array[28] = std::max(array[20], array[28]);
    array[20] = tmp;
    tmp = std::min(array[21], array[29]);
    array[29] = std::max(array[21], array[29]);
    array[21] = tmp;
    tmp = std::min(array[22], array[30]);
    array[30] = std::max(array[22], array[30]);
    array[22] = tmp;
    tmp = std::min(array[23], array[31]);
    array[31] = std::max(array[23], array[31]);
    array[23] = tmp;
    tmp = std::min(array[32], array[40]);
    array[40] = std::max(array[32], array[40]);
    array[32] = tmp;
    tmp = std::min(array[33], array[41]);
    array[41] = std::max(array[33], array[41]);
    array[33] = tmp;
    tmp = std::min(array[34], array[42]);
    array[42] = std::max(array[34], array[42]);
    array[34] = tmp;
    tmp = std::min(array[35], array[43]);
    array[43] = std::max(array[35], array[43]);
    array[35] = tmp;
    tmp = std::min(array[36], array[44]);
    array[44] = std::max(array[36], array[44]);
    array[36] = tmp;
    tmp = std::min(array[37], array[45]);
    array[45] = std::max(array[37], array[45]);
    array[37] = tmp;
    tmp = std::min(array[38], array[46]);
    array[46] = std::max(array[38], array[46]);
    array[38] = tmp;
    tmp = std::min(array[39], array[47]);
    array[47] = std::max(array[39], array[47]);
    array[39] = tmp;
    tmp = std::min(array[8], array[32]);
    array[32] = std::max(array[8], array[32]);
    array[8] = tmp;
    tmp = std::min(array[9], array[33]);
    array[33] = std::max(array[9], array[33]);
    array[9] = tmp;
    tmp = std::min(array[10], array[34]);
    array[34] = std::max(array[10], array[34]);
    array[10] = tmp;
    tmp = std::min(array[11], array[35]);
    array[35] = std::max(array[11], array[35]);
    array[11] = tmp;
    tmp = std::min(array[12], array[36]);
    array[36] = std::max(array[12], array[36]);
    array[12] = tmp;
    tmp = std::min(array[13], array[37]);
    array[37] = std::max(array[13], array[37]);
    array[13] = tmp;
    tmp = std::min(array[14], array[38]);
    array[38] = std::max(array[14], array[38]);
    array[14] = tmp;
    tmp = std::min(array[15], array[39]);
    array[39] = std::max(array[15], array[39]);
    array[15] = tmp;
    tmp = std::min(array[24], array[48]);
    array[48] = std::max(array[24], array[48]);
    array[24] = tmp;
    tmp = std::min(array[8], array[16]);
    array[16] = std::max(array[8], array[16]);
    array[8] = tmp;
    tmp = std::min(array[9], array[17]);
    array[17] = std::max(array[9], array[17]);
    array[9] = tmp;
    tmp = std::min(array[10], array[18]);
    array[18] = std::max(array[10], array[18]);
    array[10] = tmp;
    tmp = std::min(array[11], array[19]);
    array[19] = std::max(array[11], array[19]);
    array[11] = tmp;
    tmp = std::min(array[12], array[20]);
    array[20] = std::max(array[12], array[20]);
    array[12] = tmp;
    tmp = std::min(array[13], array[21]);
    array[21] = std::max(array[13], array[21]);
    array[13] = tmp;
    tmp = std::min(array[14], array[22]);
    array[22] = std::max(array[14], array[22]);
    array[14] = tmp;
    tmp = std::min(array[15], array[23]);
    array[23] = std::max(array[15], array[23]);
    array[15] = tmp;
    tmp = std::min(array[24], array[32]);
    array[32] = std::max(array[24], array[32]);
    array[24] = tmp;
    tmp = std::min(array[25], array[33]);
    array[33] = std::max(array[25], array[33]);
    array[25] = tmp;
    tmp = std::min(array[26], array[34]);
    array[34] = std::max(array[26], array[34]);
    array[26] = tmp;
    tmp = std::min(array[27], array[35]);
    array[35] = std::max(array[27], array[35]);
    array[27] = tmp;
    tmp = std::min(array[28], array[36]);
    array[36] = std::max(array[28], array[36]);
    array[28] = tmp;
    tmp = std::min(array[29], array[37]);
    array[37] = std::max(array[29], array[37]);
    array[29] = tmp;
    tmp = std::min(array[30], array[38]);
    array[38] = std::max(array[30], array[38]);
    array[30] = tmp;
    tmp = std::min(array[31], array[39]);
    array[39] = std::max(array[31], array[39]);
    array[31] = tmp;
    tmp = std::min(array[40], array[48]);
    array[48] = std::max(array[40], array[48]);
    array[40] = tmp;
    tmp = std::min(array[0], array[4]);
    array[4] = std::max(array[0], array[4]);
    array[0] = tmp;
    tmp = std::min(array[1], array[5]);
    array[5] = std::max(array[1], array[5]);
    array[1] = tmp;
    tmp = std::min(array[2], array[6]);
    array[6] = std::max(array[2], array[6]);
    array[2] = tmp;
    tmp = std::min(array[3], array[7]);
    array[7] = std::max(array[3], array[7]);
    array[3] = tmp;
    tmp = std::min(array[8], array[12]);
    array[12] = std::max(array[8], array[12]);
    array[8] = tmp;
    tmp = std::min(array[9], array[13]);
    array[13] = std::max(array[9], array[13]);
    array[9] = tmp;
    tmp = std::min(array[10], array[14]);
    array[14] = std::max(array[10], array[14]);
    array[10] = tmp;
    tmp = std::min(array[11], array[15]);
    array[15] = std::max(array[11], array[15]);
    array[11] = tmp;
    tmp = std::min(array[16], array[20]);
    array[20] = std::max(array[16], array[20]);
    array[16] = tmp;
    tmp = std::min(array[17], array[21]);
    array[21] = std::max(array[17], array[21]);
    array[17] = tmp;
    tmp = std::min(array[18], array[22]);
    array[22] = std::max(array[18], array[22]);
    array[18] = tmp;
    tmp = std::min(array[19], array[23]);
    array[23] = std::max(array[19], array[23]);
    array[19] = tmp;
    tmp = std::min(array[24], array[28]);
    array[28] = std::max(array[24], array[28]);
    array[24] = tmp;
    tmp = std::min(array[25], array[29]);
    array[29] = std::max(array[25], array[29]);
    array[25] = tmp;
    tmp = std::min(array[26], array[30]);
    array[30] = std::max(array[26], array[30]);
    array[26] = tmp;
    tmp = std::min(array[27], array[31]);
    array[31] = std::max(array[27], array[31]);
    array[27] = tmp;
    tmp = std::min(array[32], array[36]);
    array[36] = std::max(array[32], array[36]);
    array[32] = tmp;
    tmp = std::min(array[33], array[37]);
    array[37] = std::max(array[33], array[37]);
    array[33] = tmp;
    tmp = std::min(array[34], array[38]);
    array[38] = std::max(array[34], array[38]);
    array[34] = tmp;
    tmp = std::min(array[35], array[39]);
    array[39] = std::max(array[35], array[39]);
    array[35] = tmp;
    tmp = std::min(array[40], array[44]);
    array[44] = std::max(array[40], array[44]);
    array[40] = tmp;
    tmp = std::min(array[41], array[45]);
    array[45] = std::max(array[41], array[45]);
    array[41] = tmp;
    tmp = std::min(array[42], array[46]);
    array[46] = std::max(array[42], array[46]);
    array[42] = tmp;
    tmp = std::min(array[43], array[47]);
    array[47] = std::max(array[43], array[47]);
    array[43] = tmp;
    tmp = std::min(array[4], array[32]);
    array[32] = std::max(array[4], array[32]);
    array[4] = tmp;
    tmp = std::min(array[5], array[33]);
    array[33] = std::max(array[5], array[33]);
    array[5] = tmp;
    tmp = std::min(array[6], array[34]);
    array[34] = std::max(array[6], array[34]);
    array[6] = tmp;
    tmp = std::min(array[7], array[35]);
    array[35] = std::max(array[7], array[35]);
    array[7] = tmp;
    tmp = std::min(array[12], array[40]);
    array[40] = std::max(array[12], array[40]);
    array[12] = tmp;
    tmp = std::min(array[13], array[41]);
    array[41] = std::max(array[13], array[41]);
    array[13] = tmp;
    tmp = std::min(array[14], array[42]);
    array[42] = std::max(array[14], array[42]);
    array[14] = tmp;
    tmp = std::min(array[15], array[43]);
    array[43] = std::max(array[15], array[43]);
    array[15] = tmp;
    tmp = std::min(array[20], array[48]);
    array[48] = std::max(array[20], array[48]);
    array[20] = tmp;
    tmp = std::min(array[4], array[16]);
    array[16] = std::max(array[4], array[16]);
    array[4] = tmp;
    tmp = std::min(array[5], array[17]);
    array[17] = std::max(array[5], array[17]);
    array[5] = tmp;
    tmp = std::min(array[6], array[18]);
    array[18] = std::max(array[6], array[18]);
    array[6] = tmp;
    tmp = std::min(array[7], array[19]);
    array[19] = std::max(array[7], array[19]);
    array[7] = tmp;
    tmp = std::min(array[12], array[24]);
    array[24] = std::max(array[12], array[24]);
    array[12] = tmp;
    tmp = std::min(array[13], array[25]);
    array[25] = std::max(array[13], array[25]);
    array[13] = tmp;
    tmp = std::min(array[14], array[26]);
    array[26] = std::max(array[14], array[26]);
    array[14] = tmp;
    tmp = std::min(array[15], array[27]);
    array[27] = std::max(array[15], array[27]);
    array[15] = tmp;
    tmp = std::min(array[20], array[32]);
    array[32] = std::max(array[20], array[32]);
    array[20] = tmp;
    tmp = std::min(array[21], array[33]);
    array[33] = std::max(array[21], array[33]);
    array[21] = tmp;
    tmp = std::min(array[22], array[34]);
    array[34] = std::max(array[22], array[34]);
    array[22] = tmp;
    tmp = std::min(array[23], array[35]);
    array[35] = std::max(array[23], array[35]);
    array[23] = tmp;
    tmp = std::min(array[28], array[40]);
    array[40] = std::max(array[28], array[40]);
    array[28] = tmp;
    tmp = std::min(array[29], array[41]);
    array[41] = std::max(array[29], array[41]);
    array[29] = tmp;
    tmp = std::min(array[30], array[42]);
    array[42] = std::max(array[30], array[42]);
    array[30] = tmp;
    tmp = std::min(array[31], array[43]);
    array[43] = std::max(array[31], array[43]);
    array[31] = tmp;
    tmp = std::min(array[36], array[48]);
    array[48] = std::max(array[36], array[48]);
    array[36] = tmp;
    tmp = std::min(array[4], array[8]);
    array[8] = std::max(array[4], array[8]);
    array[4] = tmp;
    tmp = std::min(array[5], array[9]);
    array[9] = std::max(array[5], array[9]);
    array[5] = tmp;
    tmp = std::min(array[6], array[10]);
    array[10] = std::max(array[6], array[10]);
    array[6] = tmp;
    tmp = std::min(array[7], array[11]);
    array[11] = std::max(array[7], array[11]);
    array[7] = tmp;
    tmp = std::min(array[12], array[16]);
    array[16] = std::max(array[12], array[16]);
    array[12] = tmp;
    tmp = std::min(array[13], array[17]);
    array[17] = std::max(array[13], array[17]);
    array[13] = tmp;
    tmp = std::min(array[14], array[18]);
    array[18] = std::max(array[14], array[18]);
    array[14] = tmp;
    tmp = std::min(array[15], array[19]);
    array[19] = std::max(array[15], array[19]);
    array[15] = tmp;
    tmp = std::min(array[20], array[24]);
    array[24] = std::max(array[20], array[24]);
    array[20] = tmp;
    tmp = std::min(array[21], array[25]);
    array[25] = std::max(array[21], array[25]);
    array[21] = tmp;
    tmp = std::min(array[22], array[26]);
    array[26] = std::max(array[22], array[26]);
    array[22] = tmp;
    tmp = std::min(array[23], array[27]);
    array[27] = std::max(array[23], array[27]);
    array[23] = tmp;
    tmp = std::min(array[28], array[32]);
    array[32] = std::max(array[28], array[32]);
    array[28] = tmp;
    tmp = std::min(array[29], array[33]);
    array[33] = std::max(array[29], array[33]);
    array[29] = tmp;
    tmp = std::min(array[30], array[34]);
    array[34] = std::max(array[30], array[34]);
    array[30] = tmp;
    tmp = std::min(array[31], array[35]);
    array[35] = std::max(array[31], array[35]);
    array[31] = tmp;
    tmp = std::min(array[36], array[40]);
    array[40] = std::max(array[36], array[40]);
    array[36] = tmp;
    tmp = std::min(array[37], array[41]);
    array[41] = std::max(array[37], array[41]);
    array[37] = tmp;
    tmp = std::min(array[38], array[42]);
    array[42] = std::max(array[38], array[42]);
    array[38] = tmp;
    tmp = std::min(array[39], array[43]);
    array[43] = std::max(array[39], array[43]);
    array[39] = tmp;
    tmp = std::min(array[44], array[48]);
    array[48] = std::max(array[44], array[48]);
    array[44] = tmp;
    tmp = std::min(array[0], array[2]);
    array[2] = std::max(array[0], array[2]);
    array[0] = tmp;
    tmp = std::min(array[1], array[3]);
    array[3] = std::max(array[1], array[3]);
    array[1] = tmp;
    tmp = std::min(array[4], array[6]);
    array[6] = std::max(array[4], array[6]);
    array[4] = tmp;
    tmp = std::min(array[5], array[7]);
    array[7] = std::max(array[5], array[7]);
    array[5] = tmp;
    tmp = std::min(array[8], array[10]);
    array[10] = std::max(array[8], array[10]);
    array[8] = tmp;
    tmp = std::min(array[9], array[11]);
    array[11] = std::max(array[9], array[11]);
    array[9] = tmp;
    tmp = std::min(array[12], array[14]);
    array[14] = std::max(array[12], array[14]);
    array[12] = tmp;
    tmp = std::min(array[13], array[15]);
    array[15] = std::max(array[13], array[15]);
    array[13] = tmp;
    tmp = std::min(array[16], array[18]);
    array[18] = std::max(array[16], array[18]);
    array[16] = tmp;
    tmp = std::min(array[17], array[19]);
    array[19] = std::max(array[17], array[19]);
    array[17] = tmp;
    tmp = std::min(array[20], array[22]);
    array[22] = std::max(array[20], array[22]);
    array[20] = tmp;
    tmp = std::min(array[21], array[23]);
    array[23] = std::max(array[21], array[23]);
    array[21] = tmp;
    tmp = std::min(array[24], array[26]);
    array[26] = std::max(array[24], array[26]);
    array[24] = tmp;
    tmp = std::min(array[25], array[27]);
    array[27] = std::max(array[25], array[27]);
    array[25] = tmp;
    tmp = std::min(array[28], array[30]);
    array[30] = std::max(array[28], array[30]);
    array[28] = tmp;
    tmp = std::min(array[29], array[31]);
    array[31] = std::max(array[29], array[31]);
    array[29] = tmp;
    tmp = std::min(array[32], array[34]);
    array[34] = std::max(array[32], array[34]);
    array[32] = tmp;
    tmp = std::min(array[33], array[35]);
    array[35] = std::max(array[33], array[35]);
    array[33] = tmp;
    tmp = std::min(array[36], array[38]);
    array[38] = std::max(array[36], array[38]);
    array[36] = tmp;
    tmp = std::min(array[37], array[39]);
    array[39] = std::max(array[37], array[39]);
    array[37] = tmp;
    tmp = std::min(array[40], array[42]);
    array[42] = std::max(array[40], array[42]);
    array[40] = tmp;
    tmp = std::min(array[41], array[43]);
    array[43] = std::max(array[41], array[43]);
    array[41] = tmp;
    tmp = std::min(array[44], array[46]);
    array[46] = std::max(array[44], array[46]);
    array[44] = tmp;
    tmp = std::min(array[45], array[47]);
    array[47] = std::max(array[45], array[47]);
    array[45] = tmp;
    tmp = std::min(array[2], array[32]);
    array[32] = std::max(array[2], array[32]);
    array[2] = tmp;
    tmp = std::min(array[3], array[33]);
    array[33] = std::max(array[3], array[33]);
    array[3] = tmp;
    tmp = std::min(array[6], array[36]);
    array[36] = std::max(array[6], array[36]);
    array[6] = tmp;
    tmp = std::min(array[7], array[37]);
    array[37] = std::max(array[7], array[37]);
    array[7] = tmp;
    tmp = std::min(array[10], array[40]);
    array[40] = std::max(array[10], array[40]);
    array[10] = tmp;
    tmp = std::min(array[11], array[41]);
    array[41] = std::max(array[11], array[41]);
    array[11] = tmp;
    tmp = std::min(array[14], array[44]);
    array[44] = std::max(array[14], array[44]);
    array[14] = tmp;
    tmp = std::min(array[15], array[45]);
    array[45] = std::max(array[15], array[45]);
    array[15] = tmp;
    tmp = std::min(array[18], array[48]);
    array[48] = std::max(array[18], array[48]);
    array[18] = tmp;
    tmp = std::min(array[2], array[16]);
    array[16] = std::max(array[2], array[16]);
    array[2] = tmp;
    tmp = std::min(array[3], array[17]);
    array[17] = std::max(array[3], array[17]);
    array[3] = tmp;
    tmp = std::min(array[6], array[20]);
    array[20] = std::max(array[6], array[20]);
    array[6] = tmp;
    tmp = std::min(array[7], array[21]);
    array[21] = std::max(array[7], array[21]);
    array[7] = tmp;
    tmp = std::min(array[10], array[24]);
    array[24] = std::max(array[10], array[24]);
    array[10] = tmp;
    tmp = std::min(array[11], array[25]);
    array[25] = std::max(array[11], array[25]);
    array[11] = tmp;
    tmp = std::min(array[14], array[28]);
    array[28] = std::max(array[14], array[28]);
    array[14] = tmp;
    tmp = std::min(array[15], array[29]);
    array[29] = std::max(array[15], array[29]);
    array[15] = tmp;
    tmp = std::min(array[18], array[32]);
    array[32] = std::max(array[18], array[32]);
    array[18] = tmp;
    tmp = std::min(array[19], array[33]);
    array[33] = std::max(array[19], array[33]);
    array[19] = tmp;
    tmp = std::min(array[22], array[36]);
    array[36] = std::max(array[22], array[36]);
    array[22] = tmp;
    tmp = std::min(array[23], array[37]);
    array[37] = std::max(array[23], array[37]);
    array[23] = tmp;
    tmp = std::min(array[26], array[40]);
    array[40] = std::max(array[26], array[40]);
    array[26] = tmp;
    tmp = std::min(array[27], array[41]);
    array[41] = std::max(array[27], array[41]);
    array[27] = tmp;
    tmp = std::min(array[30], array[44]);
    array[44] = std::max(array[30], array[44]);
    array[30] = tmp;
    tmp = std::min(array[31], array[45]);
    array[45] = std::max(array[31], array[45]);
    array[31] = tmp;
    tmp = std::min(array[34], array[48]);
    array[48] = std::max(array[34], array[48]);
    array[34] = tmp;
    tmp = std::min(array[2], array[8]);
    array[8] = std::max(array[2], array[8]);
    array[2] = tmp;
    tmp = std::min(array[3], array[9]);
    array[9] = std::max(array[3], array[9]);
    array[3] = tmp;
    tmp = std::min(array[6], array[12]);
    array[12] = std::max(array[6], array[12]);
    array[6] = tmp;
    tmp = std::min(array[7], array[13]);
    array[13] = std::max(array[7], array[13]);
    array[7] = tmp;
    tmp = std::min(array[10], array[16]);
    array[16] = std::max(array[10], array[16]);
    array[10] = tmp;
    tmp = std::min(array[11], array[17]);
    array[17] = std::max(array[11], array[17]);
    array[11] = tmp;
    tmp = std::min(array[14], array[20]);
    array[20] = std::max(array[14], array[20]);
    array[14] = tmp;
    tmp = std::min(array[15], array[21]);
    array[21] = std::max(array[15], array[21]);
    array[15] = tmp;
    tmp = std::min(array[18], array[24]);
    array[24] = std::max(array[18], array[24]);
    array[18] = tmp;
    tmp = std::min(array[19], array[25]);
    array[25] = std::max(array[19], array[25]);
    array[19] = tmp;
    tmp = std::min(array[22], array[28]);
    array[28] = std::max(array[22], array[28]);
    array[22] = tmp;
    tmp = std::min(array[23], array[29]);
    array[29] = std::max(array[23], array[29]);
    array[23] = tmp;
    tmp = std::min(array[26], array[32]);
    array[32] = std::max(array[26], array[32]);
    array[26] = tmp;
    tmp = std::min(array[27], array[33]);
    array[33] = std::max(array[27], array[33]);
    array[27] = tmp;
    tmp = std::min(array[30], array[36]);
    array[36] = std::max(array[30], array[36]);
    array[30] = tmp;
    tmp = std::min(array[31], array[37]);
    array[37] = std::max(array[31], array[37]);
    array[31] = tmp;
    tmp = std::min(array[34], array[40]);
    array[40] = std::max(array[34], array[40]);
    array[34] = tmp;
    tmp = std::min(array[35], array[41]);
    array[41] = std::max(array[35], array[41]);
    array[35] = tmp;
    tmp = std::min(array[38], array[44]);
    array[44] = std::max(array[38], array[44]);
    array[38] = tmp;
    tmp = std::min(array[39], array[45]);
    array[45] = std::max(array[39], array[45]);
    array[39] = tmp;
    tmp = std::min(array[42], array[48]);
    array[48] = std::max(array[42], array[48]);
    array[42] = tmp;
    tmp = std::min(array[2], array[4]);
    array[4] = std::max(array[2], array[4]);
    array[2] = tmp;
    tmp = std::min(array[3], array[5]);
    array[5] = std::max(array[3], array[5]);
    array[3] = tmp;
    tmp = std::min(array[6], array[8]);
    array[8] = std::max(array[6], array[8]);
    array[6] = tmp;
    tmp = std::min(array[7], array[9]);
    array[9] = std::max(array[7], array[9]);
    array[7] = tmp;
    tmp = std::min(array[10], array[12]);
    array[12] = std::max(array[10], array[12]);
    array[10] = tmp;
    tmp = std::min(array[11], array[13]);
    array[13] = std::max(array[11], array[13]);
    array[11] = tmp;
    tmp = std::min(array[14], array[16]);
    array[16] = std::max(array[14], array[16]);
    array[14] = tmp;
    tmp = std::min(array[15], array[17]);
    array[17] = std::max(array[15], array[17]);
    array[15] = tmp;
    tmp = std::min(array[18], array[20]);
    array[20] = std::max(array[18], array[20]);
    array[18] = tmp;
    tmp = std::min(array[19], array[21]);
    array[21] = std::max(array[19], array[21]);
    array[19] = tmp;
    tmp = std::min(array[22], array[24]);
    array[24] = std::max(array[22], array[24]);
    array[22] = tmp;
    tmp = std::min(array[23], array[25]);
    array[25] = std::max(array[23], array[25]);
    array[23] = tmp;
    tmp = std::min(array[26], array[28]);
    array[28] = std::max(array[26], array[28]);
    array[26] = tmp;
    tmp = std::min(array[27], array[29]);
    array[29] = std::max(array[27], array[29]);
    array[27] = tmp;
    tmp = std::min(array[30], array[32]);
    array[32] = std::max(array[30], array[32]);
    array[30] = tmp;
    tmp = std::min(array[31], array[33]);
    array[33] = std::max(array[31], array[33]);
    array[31] = tmp;
    tmp = std::min(array[34], array[36]);
    array[36] = std::max(array[34], array[36]);
    array[34] = tmp;
    tmp = std::min(array[35], array[37]);
    array[37] = std::max(array[35], array[37]);
    array[35] = tmp;
    tmp = std::min(array[38], array[40]);
    array[40] = std::max(array[38], array[40]);
    array[38] = tmp;
    tmp = std::min(array[39], array[41]);
    array[41] = std::max(array[39], array[41]);
    array[39] = tmp;
    tmp = std::min(array[42], array[44]);
    array[44] = std::max(array[42], array[44]);
    array[42] = tmp;
    tmp = std::min(array[43], array[45]);
    array[45] = std::max(array[43], array[45]);
    array[43] = tmp;
    tmp = std::min(array[46], array[48]);
    array[48] = std::max(array[46], array[48]);
    array[46] = tmp;
    array[1] = std::max(array[0], array[1]);
    array[3] = std::max(array[2], array[3]);
    array[5] = std::max(array[4], array[5]);
    array[7] = std::max(array[6], array[7]);
    array[9] = std::max(array[8], array[9]);
    array[11] = std::max(array[10], array[11]);
    array[13] = std::max(array[12], array[13]);
    array[15] = std::max(array[14], array[15]);
    array[17] = std::max(array[16], array[17]);
    array[19] = std::max(array[18], array[19]);
    array[21] = std::max(array[20], array[21]);
    array[23] = std::max(array[22], array[23]);
    array[24] = std::min(array[24], array[25]);
    array[26] = std::min(array[26], array[27]);
    array[28] = std::min(array[28], array[29]);
    array[30] = std::min(array[30], array[31]);
    array[32] = std::min(array[32], array[33]);
    array[34] = std::min(array[34], array[35]);
    array[36] = std::min(array[36], array[37]);
    array[38] = std::min(array[38], array[39]);
    array[40] = std::min(array[40], array[41]);
    array[42] = std::min(array[42], array[43]);
    array[44] = std::min(array[44], array[45]);
    array[46] = std::min(array[46], array[47]);
    array[32] = std::max(array[1], array[32]);
    array[34] = std::max(array[3], array[34]);
    array[36] = std::max(array[5], array[36]);
    array[38] = std::max(array[7], array[38]);
    array[9] = std::min(array[9], array[40]);
    array[11] = std::min(array[11], array[42]);
    array[13] = std::min(array[13], array[44]);
    array[15] = std::min(array[15], array[46]);
    array[17] = std::min(array[17], array[48]);
    array[24] = std::max(array[9], array[24]);
    array[26] = std::max(array[11], array[26]);
    array[28] = std::max(array[13], array[28]);
    array[30] = std::max(array[15], array[30]);
    array[17] = std::min(array[17], array[32]);
    array[19] = std::min(array[19], array[34]);
    array[21] = std::min(array[21], array[36]);
    array[23] = std::min(array[23], array[38]);
    array[24] = std::max(array[17], array[24]);
    array[26] = std::max(array[19], array[26]);
    array[21] = std::min(array[21], array[28]);
    array[23] = std::min(array[23], array[30]);
    array[24] = std::max(array[21], array[24]);
    array[23] = std::min(array[23], array[26]);
    return std::max(array[23], array[24]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 49> array)
{
    vfloat tmp = vminf(array[0], array[32]);
    array[32] = vmaxf(array[0], array[32]);
    array[0] = tmp;
    tmp = vminf(array[1], array[33]);
    array[33] = vmaxf(array[1], array[33]);
    array[1] = tmp;
    tmp = vminf(array[2], array[34]);
    array[34] = vmaxf(array[2], array[34]);
    array[2] = tmp;
    tmp = vminf(array[3], array[35]);
    array[35] = vmaxf(array[3], array[35]);
    array[3] = tmp;
    tmp = vminf(array[4], array[36]);
    array[36] = vmaxf(array[4], array[36]);
    array[4] = tmp;
    tmp = vminf(array[5], array[37]);
    array[37] = vmaxf(array[5], array[37]);
    array[5] = tmp;
    tmp = vminf(array[6], array[38]);
    array[38] = vmaxf(array[6], array[38]);
    array[6] = tmp;
    tmp = vminf(array[7], array[39]);
    array[39] = vmaxf(array[7], array[39]);
    array[7] = tmp;
    tmp = vminf(array[8], array[40]);
    array[40] = vmaxf(array[8], array[40]);
    array[8] = tmp;
    tmp = vminf(array[9], array[41]);
    array[41] = vmaxf(array[9], array[41]);
    array[9] = tmp;
    tmp = vminf(array[10], array[42]);
    array[42] = vmaxf(array[10], array[42]);
    array[10] = tmp;
    tmp = vminf(array[11], array[43]);
    array[43] = vmaxf(array[11], array[43]);
    array[11] = tmp;
    tmp = vminf(array[12], array[44]);
    array[44] = vmaxf(array[12], array[44]);
    array[12] = tmp;
    tmp = vminf(array[13], array[45]);
    array[45] = vmaxf(array[13], array[45]);
    array[13] = tmp;
    tmp = vminf(array[14], array[46]);
    array[46] = vmaxf(array[14], array[46]);
    array[14] = tmp;
    tmp = vminf(array[15], array[47]);
    array[47] = vmaxf(array[15], array[47]);
    array[15] = tmp;
    tmp = vminf(array[16], array[48]);
    array[48] = vmaxf(array[16], array[48]);
    array[16] = tmp;
    tmp = vminf(array[0], array[16]);
    array[16] = vmaxf(array[0], array[16]);
    array[0] = tmp;
    tmp = vminf(array[1], array[17]);
    array[17] = vmaxf(array[1], array[17]);
    array[1] = tmp;
    tmp = vminf(array[2], array[18]);
    array[18] = vmaxf(array[2], array[18]);
    array[2] = tmp;
    tmp = vminf(array[3], array[19]);
    array[19] = vmaxf(array[3], array[19]);
    array[3] = tmp;
    tmp = vminf(array[4], array[20]);
    array[20] = vmaxf(array[4], array[20]);
    array[4] = tmp;
    tmp = vminf(array[5], array[21]);
    array[21] = vmaxf(array[5], array[21]);
    array[5] = tmp;
    tmp = vminf(array[6], array[22]);
    array[22] = vmaxf(array[6], array[22]);
    array[6] = tmp;
    tmp = vminf(array[7], array[23]);
    array[23] = vmaxf(array[7], array[23]);
    array[7] = tmp;
    tmp = vminf(array[8], array[24]);
    array[24] = vmaxf(array[8], array[24]);
    array[8] = tmp;
    tmp = vminf(array[9], array[25]);
    array[25] = vmaxf(array[9], array[25]);
    array[9] = tmp;
    tmp = vminf(array[10], array[26]);
    array[26] = vmaxf(array[10], array[26]);
    array[10] = tmp;
    tmp = vminf(array[11], array[27]);
    array[27] = vmaxf(array[11], array[27]);
    array[11] = tmp;
    tmp = vminf(array[12], array[28]);
    array[28] = vmaxf(array[12], array[28]);
    array[12] = tmp;
    tmp = vminf(array[13], array[29]);
    array[29] = vmaxf(array[13], array[29]);
    array[13] = tmp;
    tmp = vminf(array[14], array[30]);
    array[30] = vmaxf(array[14], array[30]);
    array[14] = tmp;
    tmp = vminf(array[15], array[31]);
    array[31] = vmaxf(array[15], array[31]);
    array[15] = tmp;
    tmp = vminf(array[32], array[48]);
    array[48] = vmaxf(array[32], array[48]);
    array[32] = tmp;
    tmp = vminf(array[16], array[32]);
    array[32] = vmaxf(array[16], array[32]);
    array[16] = tmp;
    tmp = vminf(array[17], array[33]);
    array[33] = vmaxf(array[17], array[33]);
    array[17] = tmp;
    tmp = vminf(array[18], array[34]);
    array[34] = vmaxf(array[18], array[34]);
    array[18] = tmp;
    tmp = vminf(array[19], array[35]);
    array[35] = vmaxf(array[19], array[35]);
    array[19] = tmp;
    tmp = vminf(array[20], array[36]);
    array[36] = vmaxf(array[20], array[36]);
    array[20] = tmp;
    tmp = vminf(array[21], array[37]);
    array[37] = vmaxf(array[21], array[37]);
    array[21] = tmp;
    tmp = vminf(array[22], array[38]);
    array[38] = vmaxf(array[22], array[38]);
    array[22] = tmp;
    tmp = vminf(array[23], array[39]);
    array[39] = vmaxf(array[23], array[39]);
    array[23] = tmp;
    tmp = vminf(array[24], array[40]);
    array[40] = vmaxf(array[24], array[40]);
    array[24] = tmp;
    tmp = vminf(array[25], array[41]);
    array[41] = vmaxf(array[25], array[41]);
    array[25] = tmp;
    tmp = vminf(array[26], array[42]);
    array[42] = vmaxf(array[26], array[42]);
    array[26] = tmp;
    tmp = vminf(array[27], array[43]);
    array[43] = vmaxf(array[27], array[43]);
    array[27] = tmp;
    tmp = vminf(array[28], array[44]);
    array[44] = vmaxf(array[28], array[44]);
    array[28] = tmp;
    tmp = vminf(array[29], array[45]);
    array[45] = vmaxf(array[29], array[45]);
    array[29] = tmp;
    tmp = vminf(array[30], array[46]);
    array[46] = vmaxf(array[30], array[46]);
    array[30] = tmp;
    tmp = vminf(array[31], array[47]);
    array[47] = vmaxf(array[31], array[47]);
    array[31] = tmp;
    tmp = vminf(array[0], array[8]);
    array[8] = vmaxf(array[0], array[8]);
    array[0] = tmp;
    tmp = vminf(array[1], array[9]);
    array[9] = vmaxf(array[1], array[9]);
    array[1] = tmp;
    tmp = vminf(array[2], array[10]);
    array[10] = vmaxf(array[2], array[10]);
    array[2] = tmp;
    tmp = vminf(array[3], array[11]);
    array[11] = vmaxf(array[3], array[11]);
    array[3] = tmp;
    tmp = vminf(array[4], array[12]);
    array[12] = vmaxf(array[4], array[12]);
    array[4] = tmp;
    tmp = vminf(array[5], array[13]);
    array[13] = vmaxf(array[5], array[13]);
    array[5] = tmp;
    tmp = vminf(array[6], array[14]);
    array[14] = vmaxf(array[6], array[14]);
    array[6] = tmp;
    tmp = vminf(array[7], array[15]);
    array[15] = vmaxf(array[7], array[15]);
    array[7] = tmp;
    tmp = vminf(array[16], array[24]);
    array[24] = vmaxf(array[16], array[24]);
    array[16] = tmp;
    tmp = vminf(array[17], array[25]);
    array[25] = vmaxf(array[17], array[25]);
    array[17] = tmp;
    tmp = vminf(array[18], array[26]);
    array[26] = vmaxf(array[18], array[26]);
    array[18] = tmp;
    tmp = vminf(array[19], array[27]);
    array[27] = vmaxf(array[19], array[27]);
    array[19] = tmp;
    tmp = vminf(array[20], array[28]);
    array[28] = vmaxf(array[20], array[28]);
    array[20] = tmp;
    tmp = vminf(array[21], array[29]);
    array[29] = vmaxf(array[21], array[29]);
    array[21] = tmp;
    tmp = vminf(array[22], array[30]);
    array[30] = vmaxf(array[22], array[30]);
    array[22] = tmp;
    tmp = vminf(array[23], array[31]);
    array[31] = vmaxf(array[23], array[31]);
    array[23] = tmp;
    tmp = vminf(array[32], array[40]);
    array[40] = vmaxf(array[32], array[40]);
    array[32] = tmp;
    tmp = vminf(array[33], array[41]);
    array[41] = vmaxf(array[33], array[41]);
    array[33] = tmp;
    tmp = vminf(array[34], array[42]);
    array[42] = vmaxf(array[34], array[42]);
    array[34] = tmp;
    tmp = vminf(array[35], array[43]);
    array[43] = vmaxf(array[35], array[43]);
    array[35] = tmp;
    tmp = vminf(array[36], array[44]);
    array[44] = vmaxf(array[36], array[44]);
    array[36] = tmp;
    tmp = vminf(array[37], array[45]);
    array[45] = vmaxf(array[37], array[45]);
    array[37] = tmp;
    tmp = vminf(array[38], array[46]);
    array[46] = vmaxf(array[38], array[46]);
    array[38] = tmp;
    tmp = vminf(array[39], array[47]);
    array[47] = vmaxf(array[39], array[47]);
    array[39] = tmp;
    tmp = vminf(array[8], array[32]);
    array[32] = vmaxf(array[8], array[32]);
    array[8] = tmp;
    tmp = vminf(array[9], array[33]);
    array[33] = vmaxf(array[9], array[33]);
    array[9] = tmp;
    tmp = vminf(array[10], array[34]);
    array[34] = vmaxf(array[10], array[34]);
    array[10] = tmp;
    tmp = vminf(array[11], array[35]);
    array[35] = vmaxf(array[11], array[35]);
    array[11] = tmp;
    tmp = vminf(array[12], array[36]);
    array[36] = vmaxf(array[12], array[36]);
    array[12] = tmp;
    tmp = vminf(array[13], array[37]);
    array[37] = vmaxf(array[13], array[37]);
    array[13] = tmp;
    tmp = vminf(array[14], array[38]);
    array[38] = vmaxf(array[14], array[38]);
    array[14] = tmp;
    tmp = vminf(array[15], array[39]);
    array[39] = vmaxf(array[15], array[39]);
    array[15] = tmp;
    tmp = vminf(array[24], array[48]);
    array[48] = vmaxf(array[24], array[48]);
    array[24] = tmp;
    tmp = vminf(array[8], array[16]);
    array[16] = vmaxf(array[8], array[16]);
    array[8] = tmp;
    tmp = vminf(array[9], array[17]);
    array[17] = vmaxf(array[9], array[17]);
    array[9] = tmp;
    tmp = vminf(array[10], array[18]);
    array[18] = vmaxf(array[10], array[18]);
    array[10] = tmp;
    tmp = vminf(array[11], array[19]);
    array[19] = vmaxf(array[11], array[19]);
    array[11] = tmp;
    tmp = vminf(array[12], array[20]);
    array[20] = vmaxf(array[12], array[20]);
    array[12] = tmp;
    tmp = vminf(array[13], array[21]);
    array[21] = vmaxf(array[13], array[21]);
    array[13] = tmp;
    tmp = vminf(array[14], array[22]);
    array[22] = vmaxf(array[14], array[22]);
    array[14] = tmp;
    tmp = vminf(array[15], array[23]);
    array[23] = vmaxf(array[15], array[23]);
    array[15] = tmp;
    tmp = vminf(array[24], array[32]);
    array[32] = vmaxf(array[24], array[32]);
    array[24] = tmp;
    tmp = vminf(array[25], array[33]);
    array[33] = vmaxf(array[25], array[33]);
    array[25] = tmp;
    tmp = vminf(array[26], array[34]);
    array[34] = vmaxf(array[26], array[34]);
    array[26] = tmp;
    tmp = vminf(array[27], array[35]);
    array[35] = vmaxf(array[27], array[35]);
    array[27] = tmp;
    tmp = vminf(array[28], array[36]);
    array[36] = vmaxf(array[28], array[36]);
    array[28] = tmp;
    tmp = vminf(array[29], array[37]);
    array[37] = vmaxf(array[29], array[37]);
    array[29] = tmp;
    tmp = vminf(array[30], array[38]);
    array[38] = vmaxf(array[30], array[38]);
    array[30] = tmp;
    tmp = vminf(array[31], array[39]);
    array[39] = vmaxf(array[31], array[39]);
    array[31] = tmp;
    tmp = vminf(array[40], array[48]);
    array[48] = vmaxf(array[40], array[48]);
    array[40] = tmp;
    tmp = vminf(array[0], array[4]);
    array[4] = vmaxf(array[0], array[4]);
    array[0] = tmp;
    tmp = vminf(array[1], array[5]);
    array[5] = vmaxf(array[1], array[5]);
    array[1] = tmp;
    tmp = vminf(array[2], array[6]);
    array[6] = vmaxf(array[2], array[6]);
    array[2] = tmp;
    tmp = vminf(array[3], array[7]);
    array[7] = vmaxf(array[3], array[7]);
    array[3] = tmp;
    tmp = vminf(array[8], array[12]);
    array[12] = vmaxf(array[8], array[12]);
    array[8] = tmp;
    tmp = vminf(array[9], array[13]);
    array[13] = vmaxf(array[9], array[13]);
    array[9] = tmp;
    tmp = vminf(array[10], array[14]);
    array[14] = vmaxf(array[10], array[14]);
    array[10] = tmp;
    tmp = vminf(array[11], array[15]);
    array[15] = vmaxf(array[11], array[15]);
    array[11] = tmp;
    tmp = vminf(array[16], array[20]);
    array[20] = vmaxf(array[16], array[20]);
    array[16] = tmp;
    tmp = vminf(array[17], array[21]);
    array[21] = vmaxf(array[17], array[21]);
    array[17] = tmp;
    tmp = vminf(array[18], array[22]);
    array[22] = vmaxf(array[18], array[22]);
    array[18] = tmp;
    tmp = vminf(array[19], array[23]);
    array[23] = vmaxf(array[19], array[23]);
    array[19] = tmp;
    tmp = vminf(array[24], array[28]);
    array[28] = vmaxf(array[24], array[28]);
    array[24] = tmp;
    tmp = vminf(array[25], array[29]);
    array[29] = vmaxf(array[25], array[29]);
    array[25] = tmp;
    tmp = vminf(array[26], array[30]);
    array[30] = vmaxf(array[26], array[30]);
    array[26] = tmp;
    tmp = vminf(array[27], array[31]);
    array[31] = vmaxf(array[27], array[31]);
    array[27] = tmp;
    tmp = vminf(array[32], array[36]);
    array[36] = vmaxf(array[32], array[36]);
    array[32] = tmp;
    tmp = vminf(array[33], array[37]);
    array[37] = vmaxf(array[33], array[37]);
    array[33] = tmp;
    tmp = vminf(array[34], array[38]);
    array[38] = vmaxf(array[34], array[38]);
    array[34] = tmp;
    tmp = vminf(array[35], array[39]);
    array[39] = vmaxf(array[35], array[39]);
    array[35] = tmp;
    tmp = vminf(array[40], array[44]);
    array[44] = vmaxf(array[40], array[44]);
    array[40] = tmp;
    tmp = vminf(array[41], array[45]);
    array[45] = vmaxf(array[41], array[45]);
    array[41] = tmp;
    tmp = vminf(array[42], array[46]);
    array[46] = vmaxf(array[42], array[46]);
    array[42] = tmp;
    tmp = vminf(array[43], array[47]);
    array[47] = vmaxf(array[43], array[47]);
    array[43] = tmp;
    tmp = vminf(array[4], array[32]);
    array[32] = vmaxf(array[4], array[32]);
    array[4] = tmp;
    tmp = vminf(array[5], array[33]);
    array[33] = vmaxf(array[5], array[33]);
    array[5] = tmp;
    tmp = vminf(array[6], array[34]);
    array[34] = vmaxf(array[6], array[34]);
    array[6] = tmp;
    tmp = vminf(array[7], array[35]);
    array[35] = vmaxf(array[7], array[35]);
    array[7] = tmp;
    tmp = vminf(array[12], array[40]);
    array[40] = vmaxf(array[12], array[40]);
    array[12] = tmp;
    tmp = vminf(array[13], array[41]);
    array[41] = vmaxf(array[13], array[41]);
    array[13] = tmp;
    tmp = vminf(array[14], array[42]);
    array[42] = vmaxf(array[14], array[42]);
    array[14] = tmp;
    tmp = vminf(array[15], array[43]);
    array[43] = vmaxf(array[15], array[43]);
    array[15] = tmp;
    tmp = vminf(array[20], array[48]);
    array[48] = vmaxf(array[20], array[48]);
    array[20] = tmp;
    tmp = vminf(array[4], array[16]);
    array[16] = vmaxf(array[4], array[16]);
    array[4] = tmp;
    tmp = vminf(array[5], array[17]);
    array[17] = vmaxf(array[5], array[17]);
    array[5] = tmp;
    tmp = vminf(array[6], array[18]);
    array[18] = vmaxf(array[6], array[18]);
    array[6] = tmp;
    tmp = vminf(array[7], array[19]);
    array[19] = vmaxf(array[7], array[19]);
    array[7] = tmp;
    tmp = vminf(array[12], array[24]);
    array[24] = vmaxf(array[12], array[24]);
    array[12] = tmp;
    tmp = vminf(array[13], array[25]);
    array[25] = vmaxf(array[13], array[25]);
    array[13] = tmp;
    tmp = vminf(array[14], array[26]);
    array[26] = vmaxf(array[14], array[26]);
    array[14] = tmp;
    tmp = vminf(array[15], array[27]);
    array[27] = vmaxf(array[15], array[27]);
    array[15] = tmp;
    tmp = vminf(array[20], array[32]);
    array[32] = vmaxf(array[20], array[32]);
    array[20] = tmp;
    tmp = vminf(array[21], array[33]);
    array[33] = vmaxf(array[21], array[33]);
    array[21] = tmp;
    tmp = vminf(array[22], array[34]);
    array[34] = vmaxf(array[22], array[34]);
    array[22] = tmp;
    tmp = vminf(array[23], array[35]);
    array[35] = vmaxf(array[23], array[35]);
    array[23] = tmp;
    tmp = vminf(array[28], array[40]);
    array[40] = vmaxf(array[28], array[40]);
    array[28] = tmp;
    tmp = vminf(array[29], array[41]);
    array[41] = vmaxf(array[29], array[41]);
    array[29] = tmp;
    tmp = vminf(array[30], array[42]);
    array[42] = vmaxf(array[30], array[42]);
    array[30] = tmp;
    tmp = vminf(array[31], array[43]);
    array[43] = vmaxf(array[31], array[43]);
    array[31] = tmp;
    tmp = vminf(array[36], array[48]);
    array[48] = vmaxf(array[36], array[48]);
    array[36] = tmp;
    tmp = vminf(array[4], array[8]);
    array[8] = vmaxf(array[4], array[8]);
    array[4] = tmp;
    tmp = vminf(array[5], array[9]);
    array[9] = vmaxf(array[5], array[9]);
    array[5] = tmp;
    tmp = vminf(array[6], array[10]);
    array[10] = vmaxf(array[6], array[10]);
    array[6] = tmp;
    tmp = vminf(array[7], array[11]);
    array[11] = vmaxf(array[7], array[11]);
    array[7] = tmp;
    tmp = vminf(array[12], array[16]);
    array[16] = vmaxf(array[12], array[16]);
    array[12] = tmp;
    tmp = vminf(array[13], array[17]);
    array[17] = vmaxf(array[13], array[17]);
    array[13] = tmp;
    tmp = vminf(array[14], array[18]);
    array[18] = vmaxf(array[14], array[18]);
    array[14] = tmp;
    tmp = vminf(array[15], array[19]);
    array[19] = vmaxf(array[15], array[19]);
    array[15] = tmp;
    tmp = vminf(array[20], array[24]);
    array[24] = vmaxf(array[20], array[24]);
    array[20] = tmp;
    tmp = vminf(array[21], array[25]);
    array[25] = vmaxf(array[21], array[25]);
    array[21] = tmp;
    tmp = vminf(array[22], array[26]);
    array[26] = vmaxf(array[22], array[26]);
    array[22] = tmp;
    tmp = vminf(array[23], array[27]);
    array[27] = vmaxf(array[23], array[27]);
    array[23] = tmp;
    tmp = vminf(array[28], array[32]);
    array[32] = vmaxf(array[28], array[32]);
    array[28] = tmp;
    tmp = vminf(array[29], array[33]);
    array[33] = vmaxf(array[29], array[33]);
    array[29] = tmp;
    tmp = vminf(array[30], array[34]);
    array[34] = vmaxf(array[30], array[34]);
    array[30] = tmp;
    tmp = vminf(array[31], array[35]);
    array[35] = vmaxf(array[31], array[35]);
    array[31] = tmp;
    tmp = vminf(array[36], array[40]);
    array[40] = vmaxf(array[36], array[40]);
    array[36] = tmp;
    tmp = vminf(array[37], array[41]);
    array[41] = vmaxf(array[37], array[41]);
    array[37] = tmp;
    tmp = vminf(array[38], array[42]);
    array[42] = vmaxf(array[38], array[42]);
    array[38] = tmp;
    tmp = vminf(array[39], array[43]);
    array[43] = vmaxf(array[39], array[43]);
    array[39] = tmp;
    tmp = vminf(array[44], array[48]);
    array[48] = vmaxf(array[44], array[48]);
    array[44] = tmp;
    tmp = vminf(array[0], array[2]);
    array[2] = vmaxf(array[0], array[2]);
    array[0] = tmp;
    tmp = vminf(array[1], array[3]);
    array[3] = vmaxf(array[1], array[3]);
    array[1] = tmp;
    tmp = vminf(array[4], array[6]);
    array[6] = vmaxf(array[4], array[6]);
    array[4] = tmp;
    tmp = vminf(array[5], array[7]);
    array[7] = vmaxf(array[5], array[7]);
    array[5] = tmp;
    tmp = vminf(array[8], array[10]);
    array[10] = vmaxf(array[8], array[10]);
    array[8] = tmp;
    tmp = vminf(array[9], array[11]);
    array[11] = vmaxf(array[9], array[11]);
    array[9] = tmp;
    tmp = vminf(array[12], array[14]);
    array[14] = vmaxf(array[12], array[14]);
    array[12] = tmp;
    tmp = vminf(array[13], array[15]);
    array[15] = vmaxf(array[13], array[15]);
    array[13] = tmp;
    tmp = vminf(array[16], array[18]);
    array[18] = vmaxf(array[16], array[18]);
    array[16] = tmp;
    tmp = vminf(array[17], array[19]);
    array[19] = vmaxf(array[17], array[19]);
    array[17] = tmp;
    tmp = vminf(array[20], array[22]);
    array[22] = vmaxf(array[20], array[22]);
    array[20] = tmp;
    tmp = vminf(array[21], array[23]);
    array[23] = vmaxf(array[21], array[23]);
    array[21] = tmp;
    tmp = vminf(array[24], array[26]);
    array[26] = vmaxf(array[24], array[26]);
    array[24] = tmp;
    tmp = vminf(array[25], array[27]);
    array[27] = vmaxf(array[25], array[27]);
    array[25] = tmp;
    tmp = vminf(array[28], array[30]);
    array[30] = vmaxf(array[28], array[30]);
    array[28] = tmp;
    tmp = vminf(array[29], array[31]);
    array[31] = vmaxf(array[29], array[31]);
    array[29] = tmp;
    tmp = vminf(array[32], array[34]);
    array[34] = vmaxf(array[32], array[34]);
    array[32] = tmp;
    tmp = vminf(array[33], array[35]);
    array[35] = vmaxf(array[33], array[35]);
    array[33] = tmp;
    tmp = vminf(array[36], array[38]);
    array[38] = vmaxf(array[36], array[38]);
    array[36] = tmp;
    tmp = vminf(array[37], array[39]);
    array[39] = vmaxf(array[37], array[39]);
    array[37] = tmp;
    tmp = vminf(array[40], array[42]);
    array[42] = vmaxf(array[40], array[42]);
    array[40] = tmp;
    tmp = vminf(array[41], array[43]);
    array[43] = vmaxf(array[41], array[43]);
    array[41] = tmp;
    tmp = vminf(array[44], array[46]);
    array[46] = vmaxf(array[44], array[46]);
    array[44] = tmp;
    tmp = vminf(array[45], array[47]);
    array[47] = vmaxf(array[45], array[47]);
    array[45] = tmp;
    tmp = vminf(array[2], array[32]);
    array[32] = vmaxf(array[2], array[32]);
    array[2] = tmp;
    tmp = vminf(array[3], array[33]);
    array[33] = vmaxf(array[3], array[33]);
    array[3] = tmp;
    tmp = vminf(array[6], array[36]);
    array[36] = vmaxf(array[6], array[36]);
    array[6] = tmp;
    tmp = vminf(array[7], array[37]);
    array[37] = vmaxf(array[7], array[37]);
    array[7] = tmp;
    tmp = vminf(array[10], array[40]);
    array[40] = vmaxf(array[10], array[40]);
    array[10] = tmp;
    tmp = vminf(array[11], array[41]);
    array[41] = vmaxf(array[11], array[41]);
    array[11] = tmp;
    tmp = vminf(array[14], array[44]);
    array[44] = vmaxf(array[14], array[44]);
    array[14] = tmp;
    tmp = vminf(array[15], array[45]);
    array[45] = vmaxf(array[15], array[45]);
    array[15] = tmp;
    tmp = vminf(array[18], array[48]);
    array[48] = vmaxf(array[18], array[48]);
    array[18] = tmp;
    tmp = vminf(array[2], array[16]);
    array[16] = vmaxf(array[2], array[16]);
    array[2] = tmp;
    tmp = vminf(array[3], array[17]);
    array[17] = vmaxf(array[3], array[17]);
    array[3] = tmp;
    tmp = vminf(array[6], array[20]);
    array[20] = vmaxf(array[6], array[20]);
    array[6] = tmp;
    tmp = vminf(array[7], array[21]);
    array[21] = vmaxf(array[7], array[21]);
    array[7] = tmp;
    tmp = vminf(array[10], array[24]);
    array[24] = vmaxf(array[10], array[24]);
    array[10] = tmp;
    tmp = vminf(array[11], array[25]);
    array[25] = vmaxf(array[11], array[25]);
    array[11] = tmp;
    tmp = vminf(array[14], array[28]);
    array[28] = vmaxf(array[14], array[28]);
    array[14] = tmp;
    tmp = vminf(array[15], array[29]);
    array[29] = vmaxf(array[15], array[29]);
    array[15] = tmp;
    tmp = vminf(array[18], array[32]);
    array[32] = vmaxf(array[18], array[32]);
    array[18] = tmp;
    tmp = vminf(array[19], array[33]);
    array[33] = vmaxf(array[19], array[33]);
    array[19] = tmp;
    tmp = vminf(array[22], array[36]);
    array[36] = vmaxf(array[22], array[36]);
    array[22] = tmp;
    tmp = vminf(array[23], array[37]);
    array[37] = vmaxf(array[23], array[37]);
    array[23] = tmp;
    tmp = vminf(array[26], array[40]);
    array[40] = vmaxf(array[26], array[40]);
    array[26] = tmp;
    tmp = vminf(array[27], array[41]);
    array[41] = vmaxf(array[27], array[41]);
    array[27] = tmp;
    tmp = vminf(array[30], array[44]);
    array[44] = vmaxf(array[30], array[44]);
    array[30] = tmp;
    tmp = vminf(array[31], array[45]);
    array[45] = vmaxf(array[31], array[45]);
    array[31] = tmp;
    tmp = vminf(array[34], array[48]);
    array[48] = vmaxf(array[34], array[48]);
    array[34] = tmp;
    tmp = vminf(array[2], array[8]);
    array[8] = vmaxf(array[2], array[8]);
    array[2] = tmp;
    tmp = vminf(array[3], array[9]);
    array[9] = vmaxf(array[3], array[9]);
    array[3] = tmp;
    tmp = vminf(array[6], array[12]);
    array[12] = vmaxf(array[6], array[12]);
    array[6] = tmp;
    tmp = vminf(array[7], array[13]);
    array[13] = vmaxf(array[7], array[13]);
    array[7] = tmp;
    tmp = vminf(array[10], array[16]);
    array[16] = vmaxf(array[10], array[16]);
    array[10] = tmp;
    tmp = vminf(array[11], array[17]);
    array[17] = vmaxf(array[11], array[17]);
    array[11] = tmp;
    tmp = vminf(array[14], array[20]);
    array[20] = vmaxf(array[14], array[20]);
    array[14] = tmp;
    tmp = vminf(array[15], array[21]);
    array[21] = vmaxf(array[15], array[21]);
    array[15] = tmp;
    tmp = vminf(array[18], array[24]);
    array[24] = vmaxf(array[18], array[24]);
    array[18] = tmp;
    tmp = vminf(array[19], array[25]);
    array[25] = vmaxf(array[19], array[25]);
    array[19] = tmp;
    tmp = vminf(array[22], array[28]);
    array[28] = vmaxf(array[22], array[28]);
    array[22] = tmp;
    tmp = vminf(array[23], array[29]);
    array[29] = vmaxf(array[23], array[29]);
    array[23] = tmp;
    tmp = vminf(array[26], array[32]);
    array[32] = vmaxf(array[26], array[32]);
    array[26] = tmp;
    tmp = vminf(array[27], array[33]);
    array[33] = vmaxf(array[27], array[33]);
    array[27] = tmp;
    tmp = vminf(array[30], array[36]);
    array[36] = vmaxf(array[30], array[36]);
    array[30] = tmp;
    tmp = vminf(array[31], array[37]);
    array[37] = vmaxf(array[31], array[37]);
    array[31] = tmp;
    tmp = vminf(array[34], array[40]);
    array[40] = vmaxf(array[34], array[40]);
    array[34] = tmp;
    tmp = vminf(array[35], array[41]);
    array[41] = vmaxf(array[35], array[41]);
    array[35] = tmp;
    tmp = vminf(array[38], array[44]);
    array[44] = vmaxf(array[38], array[44]);
    array[38] = tmp;
    tmp = vminf(array[39], array[45]);
    array[45] = vmaxf(array[39], array[45]);
    array[39] = tmp;
    tmp = vminf(array[42], array[48]);
    array[48] = vmaxf(array[42], array[48]);
    array[42] = tmp;
    tmp = vminf(array[2], array[4]);
    array[4] = vmaxf(array[2], array[4]);
    array[2] = tmp;
    tmp = vminf(array[3], array[5]);
    array[5] = vmaxf(array[3], array[5]);
    array[3] = tmp;
    tmp = vminf(array[6], array[8]);
    array[8] = vmaxf(array[6], array[8]);
    array[6] = tmp;
    tmp = vminf(array[7], array[9]);
    array[9] = vmaxf(array[7], array[9]);
    array[7] = tmp;
    tmp = vminf(array[10], array[12]);
    array[12] = vmaxf(array[10], array[12]);
    array[10] = tmp;
    tmp = vminf(array[11], array[13]);
    array[13] = vmaxf(array[11], array[13]);
    array[11] = tmp;
    tmp = vminf(array[14], array[16]);
    array[16] = vmaxf(array[14], array[16]);
    array[14] = tmp;
    tmp = vminf(array[15], array[17]);
    array[17] = vmaxf(array[15], array[17]);
    array[15] = tmp;
    tmp = vminf(array[18], array[20]);
    array[20] = vmaxf(array[18], array[20]);
    array[18] = tmp;
    tmp = vminf(array[19], array[21]);
    array[21] = vmaxf(array[19], array[21]);
    array[19] = tmp;
    tmp = vminf(array[22], array[24]);
    array[24] = vmaxf(array[22], array[24]);
    array[22] = tmp;
    tmp = vminf(array[23], array[25]);
    array[25] = vmaxf(array[23], array[25]);
    array[23] = tmp;
    tmp = vminf(array[26], array[28]);
    array[28] = vmaxf(array[26], array[28]);
    array[26] = tmp;
    tmp = vminf(array[27], array[29]);
    array[29] = vmaxf(array[27], array[29]);
    array[27] = tmp;
    tmp = vminf(array[30], array[32]);
    array[32] = vmaxf(array[30], array[32]);
    array[30] = tmp;
    tmp = vminf(array[31], array[33]);
    array[33] = vmaxf(array[31], array[33]);
    array[31] = tmp;
    tmp = vminf(array[34], array[36]);
    array[36] = vmaxf(array[34], array[36]);
    array[34] = tmp;
    tmp = vminf(array[35], array[37]);
    array[37] = vmaxf(array[35], array[37]);
    array[35] = tmp;
    tmp = vminf(array[38], array[40]);
    array[40] = vmaxf(array[38], array[40]);
    array[38] = tmp;
    tmp = vminf(array[39], array[41]);
    array[41] = vmaxf(array[39], array[41]);
    array[39] = tmp;
    tmp = vminf(array[42], array[44]);
    array[44] = vmaxf(array[42], array[44]);
    array[42] = tmp;
    tmp = vminf(array[43], array[45]);
    array[45] = vmaxf(array[43], array[45]);
    array[43] = tmp;
    tmp = vminf(array[46], array[48]);
    array[48] = vmaxf(array[46], array[48]);
    array[46] = tmp;
    array[1] = vmaxf(array[0], array[1]);
    array[3] = vmaxf(array[2], array[3]);
    array[5] = vmaxf(array[4], array[5]);
    array[7] = vmaxf(array[6], array[7]);
    array[9] = vmaxf(array[8], array[9]);
    array[11] = vmaxf(array[10], array[11]);
    array[13] = vmaxf(array[12], array[13]);
    array[15] = vmaxf(array[14], array[15]);
    array[17] = vmaxf(array[16], array[17]);
    array[19] = vmaxf(array[18], array[19]);
    array[21] = vmaxf(array[20], array[21]);
    array[23] = vmaxf(array[22], array[23]);
    array[24] = vminf(array[24], array[25]);
    array[26] = vminf(array[26], array[27]);
    array[28] = vminf(array[28], array[29]);
    array[30] = vminf(array[30], array[31]);
    array[32] = vminf(array[32], array[33]);
    array[34] = vminf(array[34], array[35]);
    array[36] = vminf(array[36], array[37]);
    array[38] = vminf(array[38], array[39]);
    array[40] = vminf(array[40], array[41]);
    array[42] = vminf(array[42], array[43]);
    array[44] = vminf(array[44], array[45]);
    array[46] = vminf(array[46], array[47]);
    array[32] = vmaxf(array[1], array[32]);
    array[34] = vmaxf(array[3], array[34]);
    array[36] = vmaxf(array[5], array[36]);
    array[38] = vmaxf(array[7], array[38]);
    array[9] = vminf(array[9], array[40]);
    array[11] = vminf(array[11], array[42]);
    array[13] = vminf(array[13], array[44]);
    array[15] = vminf(array[15], array[46]);
    array[17] = vminf(array[17], array[48]);
    array[24] = vmaxf(array[9], array[24]);
    array[26] = vmaxf(array[11], array[26]);
    array[28] = vmaxf(array[13], array[28]);
    array[30] = vmaxf(array[15], array[30]);
    array[17] = vminf(array[17], array[32]);
    array[19] = vminf(array[19], array[34]);
    array[21] = vminf(array[21], array[36]);
    array[23] = vminf(array[23], array[38]);
    array[24] = vmaxf(array[17], array[24]);
    array[26] = vmaxf(array[19], array[26]);
    array[21] = vminf(array[21], array[28]);
    array[23] = vminf(array[23], array[30]);
    array[24] = vmaxf(array[21], array[24]);
    array[23] = vminf(array[23], array[26]);
    return vmaxf(array[23], array[24]);
}
#endif

template<typename T>
inline T median(std::array<T, 81> array)
{
    T tmp = std::min(array[0], array[64]);
    array[64] = std::max(array[0], array[64]);
    array[0] = tmp;
    tmp = std::min(array[1], array[65]);
    array[65] = std::max(array[1], array[65]);
    array[1] = tmp;
    tmp = std::min(array[2], array[66]);
    array[66] = std::max(array[2], array[66]);
    array[2] = tmp;
    tmp = std::min(array[3], array[67]);
    array[67] = std::max(array[3], array[67]);
    array[3] = tmp;
    tmp = std::min(array[4], array[68]);
    array[68] = std::max(array[4], array[68]);
    array[4] = tmp;
    tmp = std::min(array[5], array[69]);
    array[69] = std::max(array[5], array[69]);
    array[5] = tmp;
    tmp = std::min(array[6], array[70]);
    array[70] = std::max(array[6], array[70]);
    array[6] = tmp;
    tmp = std::min(array[7], array[71]);
    array[71] = std::max(array[7], array[71]);
    array[7] = tmp;
    tmp = std::min(array[8], array[72]);
    array[72] = std::max(array[8], array[72]);
    array[8] = tmp;
    tmp = std::min(array[9], array[73]);
    array[73] = std::max(array[9], array[73]);
    array[9] = tmp;
    tmp = std::min(array[10], array[74]);
    array[74] = std::max(array[10], array[74]);
    array[10] = tmp;
    tmp = std::min(array[11], array[75]);
    array[75] = std::max(array[11], array[75]);
    array[11] = tmp;
    tmp = std::min(array[12], array[76]);
    array[76] = std::max(array[12], array[76]);
    array[12] = tmp;
    tmp = std::min(array[13], array[77]);
    array[77] = std::max(array[13], array[77]);
    array[13] = tmp;
    tmp = std::min(array[14], array[78]);
    array[78] = std::max(array[14], array[78]);
    array[14] = tmp;
    tmp = std::min(array[15], array[79]);
    array[79] = std::max(array[15], array[79]);
    array[15] = tmp;
    tmp = std::min(array[16], array[80]);
    array[80] = std::max(array[16], array[80]);
    array[16] = tmp;
    tmp = std::min(array[0], array[32]);
    array[32] = std::max(array[0], array[32]);
    array[0] = tmp;
    tmp = std::min(array[1], array[33]);
    array[33] = std::max(array[1], array[33]);
    array[1] = tmp;
    tmp = std::min(array[2], array[34]);
    array[34] = std::max(array[2], array[34]);
    array[2] = tmp;
    tmp = std::min(array[3], array[35]);
    array[35] = std::max(array[3], array[35]);
    array[3] = tmp;
    tmp = std::min(array[4], array[36]);
    array[36] = std::max(array[4], array[36]);
    array[4] = tmp;
    tmp = std::min(array[5], array[37]);
    array[37] = std::max(array[5], array[37]);
    array[5] = tmp;
    tmp = std::min(array[6], array[38]);
    array[38] = std::max(array[6], array[38]);
    array[6] = tmp;
    tmp = std::min(array[7], array[39]);
    array[39] = std::max(array[7], array[39]);
    array[7] = tmp;
    tmp = std::min(array[8], array[40]);
    array[40] = std::max(array[8], array[40]);
    array[8] = tmp;
    tmp = std::min(array[9], array[41]);
    array[41] = std::max(array[9], array[41]);
    array[9] = tmp;
    tmp = std::min(array[10], array[42]);
    array[42] = std::max(array[10], array[42]);
    array[10] = tmp;
    tmp = std::min(array[11], array[43]);
    array[43] = std::max(array[11], array[43]);
    array[11] = tmp;
    tmp = std::min(array[12], array[44]);
    array[44] = std::max(array[12], array[44]);
    array[12] = tmp;
    tmp = std::min(array[13], array[45]);
    array[45] = std::max(array[13], array[45]);
    array[13] = tmp;
    tmp = std::min(array[14], array[46]);
    array[46] = std::max(array[14], array[46]);
    array[14] = tmp;
    tmp = std::min(array[15], array[47]);
    array[47] = std::max(array[15], array[47]);
    array[15] = tmp;
    tmp = std::min(array[16], array[48]);
    array[48] = std::max(array[16], array[48]);
    array[16] = tmp;
    tmp = std::min(array[17], array[49]);
    array[49] = std::max(array[17], array[49]);
    array[17] = tmp;
    tmp = std::min(array[18], array[50]);
    array[50] = std::max(array[18], array[50]);
    array[18] = tmp;
    tmp = std::min(array[19], array[51]);
    array[51] = std::max(array[19], array[51]);
    array[19] = tmp;
    tmp = std::min(array[20], array[52]);
    array[52] = std::max(array[20], array[52]);
    array[20] = tmp;
    tmp = std::min(array[21], array[53]);
    array[53] = std::max(array[21], array[53]);
    array[21] = tmp;
    tmp = std::min(array[22], array[54]);
    array[54] = std::max(array[22], array[54]);
    array[22] = tmp;
    tmp = std::min(array[23], array[55]);
    array[55] = std::max(array[23], array[55]);
    array[23] = tmp;
    tmp = std::min(array[24], array[56]);
    array[56] = std::max(array[24], array[56]);
    array[24] = tmp;
    tmp = std::min(array[25], array[57]);
    array[57] = std::max(array[25], array[57]);
    array[25] = tmp;
    tmp = std::min(array[26], array[58]);
    array[58] = std::max(array[26], array[58]);
    array[26] = tmp;
    tmp = std::min(array[27], array[59]);
    array[59] = std::max(array[27], array[59]);
    array[27] = tmp;
    tmp = std::min(array[28], array[60]);
    array[60] = std::max(array[28], array[60]);
    array[28] = tmp;
    tmp = std::min(array[29], array[61]);
    array[61] = std::max(array[29], array[61]);
    array[29] = tmp;
    tmp = std::min(array[30], array[62]);
    array[62] = std::max(array[30], array[62]);
    array[30] = tmp;
    tmp = std::min(array[31], array[63]);
    array[63] = std::max(array[31], array[63]);
    array[31] = tmp;
    tmp = std::min(array[32], array[64]);
    array[64] = std::max(array[32], array[64]);
    array[32] = tmp;
    tmp = std::min(array[33], array[65]);
    array[65] = std::max(array[33], array[65]);
    array[33] = tmp;
    tmp = std::min(array[34], array[66]);
    array[66] = std::max(array[34], array[66]);
    array[34] = tmp;
    tmp = std::min(array[35], array[67]);
    array[67] = std::max(array[35], array[67]);
    array[35] = tmp;
    tmp = std::min(array[36], array[68]);
    array[68] = std::max(array[36], array[68]);
    array[36] = tmp;
    tmp = std::min(array[37], array[69]);
    array[69] = std::max(array[37], array[69]);
    array[37] = tmp;
    tmp = std::min(array[38], array[70]);
    array[70] = std::max(array[38], array[70]);
    array[38] = tmp;
    tmp = std::min(array[39], array[71]);
    array[71] = std::max(array[39], array[71]);
    array[39] = tmp;
    tmp = std::min(array[40], array[72]);
    array[72] = std::max(array[40], array[72]);
    array[40] = tmp;
    tmp = std::min(array[41], array[73]);
    array[73] = std::max(array[41], array[73]);
    array[41] = tmp;
    tmp = std::min(array[42], array[74]);
    array[74] = std::max(array[42], array[74]);
    array[42] = tmp;
    tmp = std::min(array[43], array[75]);
    array[75] = std::max(array[43], array[75]);
    array[43] = tmp;
    tmp = std::min(array[44], array[76]);
    array[76] = std::max(array[44], array[76]);
    array[44] = tmp;
    tmp = std::min(array[45], array[77]);
    array[77] = std::max(array[45], array[77]);
    array[45] = tmp;
    tmp = std::min(array[46], array[78]);
    array[78] = std::max(array[46], array[78]);
    array[46] = tmp;
    tmp = std::min(array[47], array[79]);
    array[79] = std::max(array[47], array[79]);
    array[47] = tmp;
    tmp = std::min(array[48], array[80]);
    array[80] = std::max(array[48], array[80]);
    array[48] = tmp;
    tmp = std::min(array[0], array[16]);
    array[16] = std::max(array[0], array[16]);
    array[0] = tmp;
    tmp = std::min(array[1], array[17]);
    array[17] = std::max(array[1], array[17]);
    array[1] = tmp;
    tmp = std::min(array[2], array[18]);
    array[18] = std::max(array[2], array[18]);
    array[2] = tmp;
    tmp = std::min(array[3], array[19]);
    array[19] = std::max(array[3], array[19]);
    array[3] = tmp;
    tmp = std::min(array[4], array[20]);
    array[20] = std::max(array[4], array[20]);
    array[4] = tmp;
    tmp = std::min(array[5], array[21]);
    array[21] = std::max(array[5], array[21]);
    array[5] = tmp;
    tmp = std::min(array[6], array[22]);
    array[22] = std::max(array[6], array[22]);
    array[6] = tmp;
    tmp = std::min(array[7], array[23]);
    array[23] = std::max(array[7], array[23]);
    array[7] = tmp;
    tmp = std::min(array[8], array[24]);
    array[24] = std::max(array[8], array[24]);
    array[8] = tmp;
    tmp = std::min(array[9], array[25]);
    array[25] = std::max(array[9], array[25]);
    array[9] = tmp;
    tmp = std::min(array[10], array[26]);
    array[26] = std::max(array[10], array[26]);
    array[10] = tmp;
    tmp = std::min(array[11], array[27]);
    array[27] = std::max(array[11], array[27]);
    array[11] = tmp;
    tmp = std::min(array[12], array[28]);
    array[28] = std::max(array[12], array[28]);
    array[12] = tmp;
    tmp = std::min(array[13], array[29]);
    array[29] = std::max(array[13], array[29]);
    array[13] = tmp;
    tmp = std::min(array[14], array[30]);
    array[30] = std::max(array[14], array[30]);
    array[14] = tmp;
    tmp = std::min(array[15], array[31]);
    array[31] = std::max(array[15], array[31]);
    array[15] = tmp;
    tmp = std::min(array[32], array[48]);
    array[48] = std::max(array[32], array[48]);
    array[32] = tmp;
    tmp = std::min(array[33], array[49]);
    array[49] = std::max(array[33], array[49]);
    array[33] = tmp;
    tmp = std::min(array[34], array[50]);
    array[50] = std::max(array[34], array[50]);
    array[34] = tmp;
    tmp = std::min(array[35], array[51]);
    array[51] = std::max(array[35], array[51]);
    array[35] = tmp;
    tmp = std::min(array[36], array[52]);
    array[52] = std::max(array[36], array[52]);
    array[36] = tmp;
    tmp = std::min(array[37], array[53]);
    array[53] = std::max(array[37], array[53]);
    array[37] = tmp;
    tmp = std::min(array[38], array[54]);
    array[54] = std::max(array[38], array[54]);
    array[38] = tmp;
    tmp = std::min(array[39], array[55]);
    array[55] = std::max(array[39], array[55]);
    array[39] = tmp;
    tmp = std::min(array[40], array[56]);
    array[56] = std::max(array[40], array[56]);
    array[40] = tmp;
    tmp = std::min(array[41], array[57]);
    array[57] = std::max(array[41], array[57]);
    array[41] = tmp;
    tmp = std::min(array[42], array[58]);
    array[58] = std::max(array[42], array[58]);
    array[42] = tmp;
    tmp = std::min(array[43], array[59]);
    array[59] = std::max(array[43], array[59]);
    array[43] = tmp;
    tmp = std::min(array[44], array[60]);
    array[60] = std::max(array[44], array[60]);
    array[44] = tmp;
    tmp = std::min(array[45], array[61]);
    array[61] = std::max(array[45], array[61]);
    array[45] = tmp;
    tmp = std::min(array[46], array[62]);
    array[62] = std::max(array[46], array[62]);
    array[46] = tmp;
    tmp = std::min(array[47], array[63]);
    array[63] = std::max(array[47], array[63]);
    array[47] = tmp;
    tmp = std::min(array[64], array[80]);
    array[80] = std::max(array[64], array[80]);
    array[64] = tmp;
    tmp = std::min(array[16], array[64]);
    array[64] = std::max(array[16], array[64]);
    array[16] = tmp;
    tmp = std::min(array[17], array[65]);
    array[65] = std::max(array[17], array[65]);
    array[17] = tmp;
    tmp = std::min(array[18], array[66]);
    array[66] = std::max(array[18], array[66]);
    array[18] = tmp;
    tmp = std::min(array[19], array[67]);
    array[67] = std::max(array[19], array[67]);
    array[19] = tmp;
    tmp = std::min(array[20], array[68]);
    array[68] = std::max(array[20], array[68]);
    array[20] = tmp;
    tmp = std::min(array[21], array[69]);
    array[69] = std::max(array[21], array[69]);
    array[21] = tmp;
    tmp = std::min(array[22], array[70]);
    array[70] = std::max(array[22], array[70]);
    array[22] = tmp;
    tmp = std::min(array[23], array[71]);
    array[71] = std::max(array[23], array[71]);
    array[23] = tmp;
    tmp = std::min(array[24], array[72]);
    array[72] = std::max(array[24], array[72]);
    array[24] = tmp;
    tmp = std::min(array[25], array[73]);
    array[73] = std::max(array[25], array[73]);
    array[25] = tmp;
    tmp = std::min(array[26], array[74]);
    array[74] = std::max(array[26], array[74]);
    array[26] = tmp;
    tmp = std::min(array[27], array[75]);
    array[75] = std::max(array[27], array[75]);
    array[27] = tmp;
    tmp = std::min(array[28], array[76]);
    array[76] = std::max(array[28], array[76]);
    array[28] = tmp;
    tmp = std::min(array[29], array[77]);
    array[77] = std::max(array[29], array[77]);
    array[29] = tmp;
    tmp = std::min(array[30], array[78]);
    array[78] = std::max(array[30], array[78]);
    array[30] = tmp;
    tmp = std::min(array[31], array[79]);
    array[79] = std::max(array[31], array[79]);
    array[31] = tmp;
    tmp = std::min(array[16], array[32]);
    array[32] = std::max(array[16], array[32]);
    array[16] = tmp;
    tmp = std::min(array[17], array[33]);
    array[33] = std::max(array[17], array[33]);
    array[17] = tmp;
    tmp = std::min(array[18], array[34]);
    array[34] = std::max(array[18], array[34]);
    array[18] = tmp;
    tmp = std::min(array[19], array[35]);
    array[35] = std::max(array[19], array[35]);
    array[19] = tmp;
    tmp = std::min(array[20], array[36]);
    array[36] = std::max(array[20], array[36]);
    array[20] = tmp;
    tmp = std::min(array[21], array[37]);
    array[37] = std::max(array[21], array[37]);
    array[21] = tmp;
    tmp = std::min(array[22], array[38]);
    array[38] = std::max(array[22], array[38]);
    array[22] = tmp;
    tmp = std::min(array[23], array[39]);
    array[39] = std::max(array[23], array[39]);
    array[23] = tmp;
    tmp = std::min(array[24], array[40]);
    array[40] = std::max(array[24], array[40]);
    array[24] = tmp;
    tmp = std::min(array[25], array[41]);
    array[41] = std::max(array[25], array[41]);
    array[25] = tmp;
    tmp = std::min(array[26], array[42]);
    array[42] = std::max(array[26], array[42]);
    array[26] = tmp;
    tmp = std::min(array[27], array[43]);
    array[43] = std::max(array[27], array[43]);
    array[27] = tmp;
    tmp = std::min(array[28], array[44]);
    array[44] = std::max(array[28], array[44]);
    array[28] = tmp;
    tmp = std::min(array[29], array[45]);
    array[45] = std::max(array[29], array[45]);
    array[29] = tmp;
    tmp = std::min(array[30], array[46]);
    array[46] = std::max(array[30], array[46]);
    array[30] = tmp;
    tmp = std::min(array[31], array[47]);
    array[47] = std::max(array[31], array[47]);
    array[31] = tmp;
    tmp = std::min(array[48], array[64]);
    array[64] = std::max(array[48], array[64]);
    array[48] = tmp;
    tmp = std::min(array[49], array[65]);
    array[65] = std::max(array[49], array[65]);
    array[49] = tmp;
    tmp = std::min(array[50], array[66]);
    array[66] = std::max(array[50], array[66]);
    array[50] = tmp;
    tmp = std::min(array[51], array[67]);
    array[67] = std::max(array[51], array[67]);
    array[51] = tmp;
    tmp = std::min(array[52], array[68]);
    array[68] = std::max(array[52], array[68]);
    array[52] = tmp;
    tmp = std::min(array[53], array[69]);
    array[69] = std::max(array[53], array[69]);
    array[53] = tmp;
    tmp = std::min(array[54], array[70]);
    array[70] = std::max(array[54], array[70]);
    array[54] = tmp;
    tmp = std::min(array[55], array[71]);
    array[71] = std::max(array[55], array[71]);
    array[55] = tmp;
    tmp = std::min(array[56], array[72]);
    array[72] = std::max(array[56], array[72]);
    array[56] = tmp;
    tmp = std::min(array[57], array[73]);
    array[73] = std::max(array[57], array[73]);
    array[57] = tmp;
    tmp = std::min(array[58], array[74]);
    array[74] = std::max(array[58], array[74]);
    array[58] = tmp;
    tmp = std::min(array[59], array[75]);
    array[75] = std::max(array[59], array[75]);
    array[59] = tmp;
    tmp = std::min(array[60], array[76]);
    array[76] = std::max(array[60], array[76]);
    array[60] = tmp;
    tmp = std::min(array[61], array[77]);
    array[77] = std::max(array[61], array[77]);
    array[61] = tmp;
    tmp = std::min(array[62], array[78]);
    array[78] = std::max(array[62], array[78]);
    array[62] = tmp;
    tmp = std::min(array[63], array[79]);
    array[79] = std::max(array[63], array[79]);
    array[63] = tmp;
    tmp = std::min(array[0], array[8]);
    array[8] = std::max(array[0], array[8]);
    array[0] = tmp;
    tmp = std::min(array[1], array[9]);
    array[9] = std::max(array[1], array[9]);
    array[1] = tmp;
    tmp = std::min(array[2], array[10]);
    array[10] = std::max(array[2], array[10]);
    array[2] = tmp;
    tmp = std::min(array[3], array[11]);
    array[11] = std::max(array[3], array[11]);
    array[3] = tmp;
    tmp = std::min(array[4], array[12]);
    array[12] = std::max(array[4], array[12]);
    array[4] = tmp;
    tmp = std::min(array[5], array[13]);
    array[13] = std::max(array[5], array[13]);
    array[5] = tmp;
    tmp = std::min(array[6], array[14]);
    array[14] = std::max(array[6], array[14]);
    array[6] = tmp;
    tmp = std::min(array[7], array[15]);
    array[15] = std::max(array[7], array[15]);
    array[7] = tmp;
    tmp = std::min(array[16], array[24]);
    array[24] = std::max(array[16], array[24]);
    array[16] = tmp;
    tmp = std::min(array[17], array[25]);
    array[25] = std::max(array[17], array[25]);
    array[17] = tmp;
    tmp = std::min(array[18], array[26]);
    array[26] = std::max(array[18], array[26]);
    array[18] = tmp;
    tmp = std::min(array[19], array[27]);
    array[27] = std::max(array[19], array[27]);
    array[19] = tmp;
    tmp = std::min(array[20], array[28]);
    array[28] = std::max(array[20], array[28]);
    array[20] = tmp;
    tmp = std::min(array[21], array[29]);
    array[29] = std::max(array[21], array[29]);
    array[21] = tmp;
    tmp = std::min(array[22], array[30]);
    array[30] = std::max(array[22], array[30]);
    array[22] = tmp;
    tmp = std::min(array[23], array[31]);
    array[31] = std::max(array[23], array[31]);
    array[23] = tmp;
    tmp = std::min(array[32], array[40]);
    array[40] = std::max(array[32], array[40]);
    array[32] = tmp;
    tmp = std::min(array[33], array[41]);
    array[41] = std::max(array[33], array[41]);
    array[33] = tmp;
    tmp = std::min(array[34], array[42]);
    array[42] = std::max(array[34], array[42]);
    array[34] = tmp;
    tmp = std::min(array[35], array[43]);
    array[43] = std::max(array[35], array[43]);
    array[35] = tmp;
    tmp = std::min(array[36], array[44]);
    array[44] = std::max(array[36], array[44]);
    array[36] = tmp;
    tmp = std::min(array[37], array[45]);
    array[45] = std::max(array[37], array[45]);
    array[37] = tmp;
    tmp = std::min(array[38], array[46]);
    array[46] = std::max(array[38], array[46]);
    array[38] = tmp;
    tmp = std::min(array[39], array[47]);
    array[47] = std::max(array[39], array[47]);
    array[39] = tmp;
    tmp = std::min(array[48], array[56]);
    array[56] = std::max(array[48], array[56]);
    array[48] = tmp;
    tmp = std::min(array[49], array[57]);
    array[57] = std::max(array[49], array[57]);
    array[49] = tmp;
    tmp = std::min(array[50], array[58]);
    array[58] = std::max(array[50], array[58]);
    array[50] = tmp;
    tmp = std::min(array[51], array[59]);
    array[59] = std::max(array[51], array[59]);
    array[51] = tmp;
    tmp = std::min(array[52], array[60]);
    array[60] = std::max(array[52], array[60]);
    array[52] = tmp;
    tmp = std::min(array[53], array[61]);
    array[61] = std::max(array[53], array[61]);
    array[53] = tmp;
    tmp = std::min(array[54], array[62]);
    array[62] = std::max(array[54], array[62]);
    array[54] = tmp;
    tmp = std::min(array[55], array[63]);
    array[63] = std::max(array[55], array[63]);
    array[55] = tmp;
    tmp = std::min(array[64], array[72]);
    array[72] = std::max(array[64], array[72]);
    array[64] = tmp;
    tmp = std::min(array[65], array[73]);
    array[73] = std::max(array[65], array[73]);
    array[65] = tmp;
    tmp = std::min(array[66], array[74]);
    array[74] = std::max(array[66], array[74]);
    array[66] = tmp;
    tmp = std::min(array[67], array[75]);
    array[75] = std::max(array[67], array[75]);
    array[67] = tmp;
    tmp = std::min(array[68], array[76]);
    array[76] = std::max(array[68], array[76]);
    array[68] = tmp;
    tmp = std::min(array[69], array[77]);
    array[77] = std::max(array[69], array[77]);
    array[69] = tmp;
    tmp = std::min(array[70], array[78]);
    array[78] = std::max(array[70], array[78]);
    array[70] = tmp;
    tmp = std::min(array[71], array[79]);
    array[79] = std::max(array[71], array[79]);
    array[71] = tmp;
    tmp = std::min(array[8], array[64]);
    array[64] = std::max(array[8], array[64]);
    array[8] = tmp;
    tmp = std::min(array[9], array[65]);
    array[65] = std::max(array[9], array[65]);
    array[9] = tmp;
    tmp = std::min(array[10], array[66]);
    array[66] = std::max(array[10], array[66]);
    array[10] = tmp;
    tmp = std::min(array[11], array[67]);
    array[67] = std::max(array[11], array[67]);
    array[11] = tmp;
    tmp = std::min(array[12], array[68]);
    array[68] = std::max(array[12], array[68]);
    array[12] = tmp;
    tmp = std::min(array[13], array[69]);
    array[69] = std::max(array[13], array[69]);
    array[13] = tmp;
    tmp = std::min(array[14], array[70]);
    array[70] = std::max(array[14], array[70]);
    array[14] = tmp;
    tmp = std::min(array[15], array[71]);
    array[71] = std::max(array[15], array[71]);
    array[15] = tmp;
    tmp = std::min(array[24], array[80]);
    array[80] = std::max(array[24], array[80]);
    array[24] = tmp;
    tmp = std::min(array[8], array[32]);
    array[32] = std::max(array[8], array[32]);
    array[8] = tmp;
    tmp = std::min(array[9], array[33]);
    array[33] = std::max(array[9], array[33]);
    array[9] = tmp;
    tmp = std::min(array[10], array[34]);
    array[34] = std::max(array[10], array[34]);
    array[10] = tmp;
    tmp = std::min(array[11], array[35]);
    array[35] = std::max(array[11], array[35]);
    array[11] = tmp;
    tmp = std::min(array[12], array[36]);
    array[36] = std::max(array[12], array[36]);
    array[12] = tmp;
    tmp = std::min(array[13], array[37]);
    array[37] = std::max(array[13], array[37]);
    array[13] = tmp;
    tmp = std::min(array[14], array[38]);
    array[38] = std::max(array[14], array[38]);
    array[14] = tmp;
    tmp = std::min(array[15], array[39]);
    array[39] = std::max(array[15], array[39]);
    array[15] = tmp;
    tmp = std::min(array[24], array[48]);
    array[48] = std::max(array[24], array[48]);
    array[24] = tmp;
    tmp = std::min(array[25], array[49]);
    array[49] = std::max(array[25], array[49]);
    array[25] = tmp;
    tmp = std::min(array[26], array[50]);
    array[50] = std::max(array[26], array[50]);
    array[26] = tmp;
    tmp = std::min(array[27], array[51]);
    array[51] = std::max(array[27], array[51]);
    array[27] = tmp;
    tmp = std::min(array[28], array[52]);
    array[52] = std::max(array[28], array[52]);
    array[28] = tmp;
    tmp = std::min(array[29], array[53]);
    array[53] = std::max(array[29], array[53]);
    array[29] = tmp;
    tmp = std::min(array[30], array[54]);
    array[54] = std::max(array[30], array[54]);
    array[30] = tmp;
    tmp = std::min(array[31], array[55]);
    array[55] = std::max(array[31], array[55]);
    array[31] = tmp;
    tmp = std::min(array[40], array[64]);
    array[64] = std::max(array[40], array[64]);
    array[40] = tmp;
    tmp = std::min(array[41], array[65]);
    array[65] = std::max(array[41], array[65]);
    array[41] = tmp;
    tmp = std::min(array[42], array[66]);
    array[66] = std::max(array[42], array[66]);
    array[42] = tmp;
    tmp = std::min(array[43], array[67]);
    array[67] = std::max(array[43], array[67]);
    array[43] = tmp;
    tmp = std::min(array[44], array[68]);
    array[68] = std::max(array[44], array[68]);
    array[44] = tmp;
    tmp = std::min(array[45], array[69]);
    array[69] = std::max(array[45], array[69]);
    array[45] = tmp;
    tmp = std::min(array[46], array[70]);
    array[70] = std::max(array[46], array[70]);
    array[46] = tmp;
    tmp = std::min(array[47], array[71]);
    array[71] = std::max(array[47], array[71]);
    array[47] = tmp;
    tmp = std::min(array[56], array[80]);
    array[80] = std::max(array[56], array[80]);
    array[56] = tmp;
    tmp = std::min(array[8], array[16]);
    array[16] = std::max(array[8], array[16]);
    array[8] = tmp;
    tmp = std::min(array[9], array[17]);
    array[17] = std::max(array[9], array[17]);
    array[9] = tmp;
    tmp = std::min(array[10], array[18]);
    array[18] = std::max(array[10], array[18]);
    array[10] = tmp;
    tmp = std::min(array[11], array[19]);
    array[19] = std::max(array[11], array[19]);
    array[11] = tmp;
    tmp = std::min(array[12], array[20]);
    array[20] = std::max(array[12], array[20]);
    array[12] = tmp;
    tmp = std::min(array[13], array[21]);
    array[21] = std::max(array[13], array[21]);
    array[13] = tmp;
    tmp = std::min(array[14], array[22]);
    array[22] = std::max(array[14], array[22]);
    array[14] = tmp;
    tmp = std::min(array[15], array[23]);
    array[23] = std::max(array[15], array[23]);
    array[15] = tmp;
    tmp = std::min(array[24], array[32]);
    array[32] = std::max(array[24], array[32]);
    array[24] = tmp;
    tmp = std::min(array[25], array[33]);
    array[33] = std::max(array[25], array[33]);
    array[25] = tmp;
    tmp = std::min(array[26], array[34]);
    array[34] = std::max(array[26], array[34]);
    array[26] = tmp;
    tmp = std::min(array[27], array[35]);
    array[35] = std::max(array[27], array[35]);
    array[27] = tmp;
    tmp = std::min(array[28], array[36]);
    array[36] = std::max(array[28], array[36]);
    array[28] = tmp;
    tmp = std::min(array[29], array[37]);
    array[37] = std::max(array[29], array[37]);
    array[29] = tmp;
    tmp = std::min(array[30], array[38]);
    array[38] = std::max(array[30], array[38]);
    array[30] = tmp;
    tmp = std::min(array[31], array[39]);
    array[39] = std::max(array[31], array[39]);
    array[31] = tmp;
    tmp = std::min(array[40], array[48]);
    array[48] = std::max(array[40], array[48]);
    array[40] = tmp;
    tmp = std::min(array[41], array[49]);
    array[49] = std::max(array[41], array[49]);
    array[41] = tmp;
    tmp = std::min(array[42], array[50]);
    array[50] = std::max(array[42], array[50]);
    array[42] = tmp;
    tmp = std::min(array[43], array[51]);
    array[51] = std::max(array[43], array[51]);
    array[43] = tmp;
    tmp = std::min(array[44], array[52]);
    array[52] = std::max(array[44], array[52]);
    array[44] = tmp;
    tmp = std::min(array[45], array[53]);
    array[53] = std::max(array[45], array[53]);
    array[45] = tmp;
    tmp = std::min(array[46], array[54]);
    array[54] = std::max(array[46], array[54]);
    array[46] = tmp;
    tmp = std::min(array[47], array[55]);
    array[55] = std::max(array[47], array[55]);
    array[47] = tmp;
    tmp = std::min(array[56], array[64]);
    array[64] = std::max(array[56], array[64]);
    array[56] = tmp;
    tmp = std::min(array[57], array[65]);
    array[65] = std::max(array[57], array[65]);
    array[57] = tmp;
    tmp = std::min(array[58], array[66]);
    array[66] = std::max(array[58], array[66]);
    array[58] = tmp;
    tmp = std::min(array[59], array[67]);
    array[67] = std::max(array[59], array[67]);
    array[59] = tmp;
    tmp = std::min(array[60], array[68]);
    array[68] = std::max(array[60], array[68]);
    array[60] = tmp;
    tmp = std::min(array[61], array[69]);
    array[69] = std::max(array[61], array[69]);
    array[61] = tmp;
    tmp = std::min(array[62], array[70]);
    array[70] = std::max(array[62], array[70]);
    array[62] = tmp;
    tmp = std::min(array[63], array[71]);
    array[71] = std::max(array[63], array[71]);
    array[63] = tmp;
    tmp = std::min(array[72], array[80]);
    array[80] = std::max(array[72], array[80]);
    array[72] = tmp;
    tmp = std::min(array[0], array[4]);
    array[4] = std::max(array[0], array[4]);
    array[0] = tmp;
    tmp = std::min(array[1], array[5]);
    array[5] = std::max(array[1], array[5]);
    array[1] = tmp;
    tmp = std::min(array[2], array[6]);
    array[6] = std::max(array[2], array[6]);
    array[2] = tmp;
    tmp = std::min(array[3], array[7]);
    array[7] = std::max(array[3], array[7]);
    array[3] = tmp;
    tmp = std::min(array[8], array[12]);
    array[12] = std::max(array[8], array[12]);
    array[8] = tmp;
    tmp = std::min(array[9], array[13]);
    array[13] = std::max(array[9], array[13]);
    array[9] = tmp;
    tmp = std::min(array[10], array[14]);
    array[14] = std::max(array[10], array[14]);
    array[10] = tmp;
    tmp = std::min(array[11], array[15]);
    array[15] = std::max(array[11], array[15]);
    array[11] = tmp;
    tmp = std::min(array[16], array[20]);
    array[20] = std::max(array[16], array[20]);
    array[16] = tmp;
    tmp = std::min(array[17], array[21]);
    array[21] = std::max(array[17], array[21]);
    array[17] = tmp;
    tmp = std::min(array[18], array[22]);
    array[22] = std::max(array[18], array[22]);
    array[18] = tmp;
    tmp = std::min(array[19], array[23]);
    array[23] = std::max(array[19], array[23]);
    array[19] = tmp;
    tmp = std::min(array[24], array[28]);
    array[28] = std::max(array[24], array[28]);
    array[24] = tmp;
    tmp = std::min(array[25], array[29]);
    array[29] = std::max(array[25], array[29]);
    array[25] = tmp;
    tmp = std::min(array[26], array[30]);
    array[30] = std::max(array[26], array[30]);
    array[26] = tmp;
    tmp = std::min(array[27], array[31]);
    array[31] = std::max(array[27], array[31]);
    array[27] = tmp;
    tmp = std::min(array[32], array[36]);
    array[36] = std::max(array[32], array[36]);
    array[32] = tmp;
    tmp = std::min(array[33], array[37]);
    array[37] = std::max(array[33], array[37]);
    array[33] = tmp;
    tmp = std::min(array[34], array[38]);
    array[38] = std::max(array[34], array[38]);
    array[34] = tmp;
    tmp = std::min(array[35], array[39]);
    array[39] = std::max(array[35], array[39]);
    array[35] = tmp;
    tmp = std::min(array[40], array[44]);
    array[44] = std::max(array[40], array[44]);
    array[40] = tmp;
    tmp = std::min(array[41], array[45]);
    array[45] = std::max(array[41], array[45]);
    array[41] = tmp;
    tmp = std::min(array[42], array[46]);
    array[46] = std::max(array[42], array[46]);
    array[42] = tmp;
    tmp = std::min(array[43], array[47]);
    array[47] = std::max(array[43], array[47]);
    array[43] = tmp;
    tmp = std::min(array[48], array[52]);
    array[52] = std::max(array[48], array[52]);
    array[48] = tmp;
    tmp = std::min(array[49], array[53]);
    array[53] = std::max(array[49], array[53]);
    array[49] = tmp;
    tmp = std::min(array[50], array[54]);
    array[54] = std::max(array[50], array[54]);
    array[50] = tmp;
    tmp = std::min(array[51], array[55]);
    array[55] = std::max(array[51], array[55]);
    array[51] = tmp;
    tmp = std::min(array[56], array[60]);
    array[60] = std::max(array[56], array[60]);
    array[56] = tmp;
    tmp = std::min(array[57], array[61]);
    array[61] = std::max(array[57], array[61]);
    array[57] = tmp;
    tmp = std::min(array[58], array[62]);
    array[62] = std::max(array[58], array[62]);
    array[58] = tmp;
    tmp = std::min(array[59], array[63]);
    array[63] = std::max(array[59], array[63]);
    array[59] = tmp;
    tmp = std::min(array[64], array[68]);
    array[68] = std::max(array[64], array[68]);
    array[64] = tmp;
    tmp = std::min(array[65], array[69]);
    array[69] = std::max(array[65], array[69]);
    array[65] = tmp;
    tmp = std::min(array[66], array[70]);
    array[70] = std::max(array[66], array[70]);
    array[66] = tmp;
    tmp = std::min(array[67], array[71]);
    array[71] = std::max(array[67], array[71]);
    array[67] = tmp;
    tmp = std::min(array[72], array[76]);
    array[76] = std::max(array[72], array[76]);
    array[72] = tmp;
    tmp = std::min(array[73], array[77]);
    array[77] = std::max(array[73], array[77]);
    array[73] = tmp;
    tmp = std::min(array[74], array[78]);
    array[78] = std::max(array[74], array[78]);
    array[74] = tmp;
    tmp = std::min(array[75], array[79]);
    array[79] = std::max(array[75], array[79]);
    array[75] = tmp;
    tmp = std::min(array[4], array[64]);
    array[64] = std::max(array[4], array[64]);
    array[4] = tmp;
    tmp = std::min(array[5], array[65]);
    array[65] = std::max(array[5], array[65]);
    array[5] = tmp;
    tmp = std::min(array[6], array[66]);
    array[66] = std::max(array[6], array[66]);
    array[6] = tmp;
    tmp = std::min(array[7], array[67]);
    array[67] = std::max(array[7], array[67]);
    array[7] = tmp;
    tmp = std::min(array[12], array[72]);
    array[72] = std::max(array[12], array[72]);
    array[12] = tmp;
    tmp = std::min(array[13], array[73]);
    array[73] = std::max(array[13], array[73]);
    array[13] = tmp;
    tmp = std::min(array[14], array[74]);
    array[74] = std::max(array[14], array[74]);
    array[14] = tmp;
    tmp = std::min(array[15], array[75]);
    array[75] = std::max(array[15], array[75]);
    array[15] = tmp;
    tmp = std::min(array[20], array[80]);
    array[80] = std::max(array[20], array[80]);
    array[20] = tmp;
    tmp = std::min(array[4], array[32]);
    array[32] = std::max(array[4], array[32]);
    array[4] = tmp;
    tmp = std::min(array[5], array[33]);
    array[33] = std::max(array[5], array[33]);
    array[5] = tmp;
    tmp = std::min(array[6], array[34]);
    array[34] = std::max(array[6], array[34]);
    array[6] = tmp;
    tmp = std::min(array[7], array[35]);
    array[35] = std::max(array[7], array[35]);
    array[7] = tmp;
    tmp = std::min(array[12], array[40]);
    array[40] = std::max(array[12], array[40]);
    array[12] = tmp;
    tmp = std::min(array[13], array[41]);
    array[41] = std::max(array[13], array[41]);
    array[13] = tmp;
    tmp = std::min(array[14], array[42]);
    array[42] = std::max(array[14], array[42]);
    array[14] = tmp;
    tmp = std::min(array[15], array[43]);
    array[43] = std::max(array[15], array[43]);
    array[15] = tmp;
    tmp = std::min(array[20], array[48]);
    array[48] = std::max(array[20], array[48]);
    array[20] = tmp;
    tmp = std::min(array[21], array[49]);
    array[49] = std::max(array[21], array[49]);
    array[21] = tmp;
    tmp = std::min(array[22], array[50]);
    array[50] = std::max(array[22], array[50]);
    array[22] = tmp;
    tmp = std::min(array[23], array[51]);
    array[51] = std::max(array[23], array[51]);
    array[23] = tmp;
    tmp = std::min(array[28], array[56]);
    array[56] = std::max(array[28], array[56]);
    array[28] = tmp;
    tmp = std::min(array[29], array[57]);
    array[57] = std::max(array[29], array[57]);
    array[29] = tmp;
    tmp = std::min(array[30], array[58]);
    array[58] = std::max(array[30], array[58]);
    array[30] = tmp;
    tmp = std::min(array[31], array[59]);
    array[59] = std::max(array[31], array[59]);
    array[31] = tmp;
    tmp = std::min(array[36], array[64]);
    array[64] = std::max(array[36], array[64]);
    array[36] = tmp;
    tmp = std::min(array[37], array[65]);
    array[65] = std::max(array[37], array[65]);
    array[37] = tmp;
    tmp = std::min(array[38], array[66]);
    array[66] = std::max(array[38], array[66]);
    array[38] = tmp;
    tmp = std::min(array[39], array[67]);
    array[67] = std::max(array[39], array[67]);
    array[39] = tmp;
    tmp = std::min(array[44], array[72]);
    array[72] = std::max(array[44], array[72]);
    array[44] = tmp;
    tmp = std::min(array[45], array[73]);
    array[73] = std::max(array[45], array[73]);
    array[45] = tmp;
    tmp = std::min(array[46], array[74]);
    array[74] = std::max(array[46], array[74]);
    array[46] = tmp;
    tmp = std::min(array[47], array[75]);
    array[75] = std::max(array[47], array[75]);
    array[47] = tmp;
    tmp = std::min(array[52], array[80]);
    array[80] = std::max(array[52], array[80]);
    array[52] = tmp;
    tmp = std::min(array[4], array[16]);
    array[16] = std::max(array[4], array[16]);
    array[4] = tmp;
    tmp = std::min(array[5], array[17]);
    array[17] = std::max(array[5], array[17]);
    array[5] = tmp;
    tmp = std::min(array[6], array[18]);
    array[18] = std::max(array[6], array[18]);
    array[6] = tmp;
    tmp = std::min(array[7], array[19]);
    array[19] = std::max(array[7], array[19]);
    array[7] = tmp;
    tmp = std::min(array[12], array[24]);
    array[24] = std::max(array[12], array[24]);
    array[12] = tmp;
    tmp = std::min(array[13], array[25]);
    array[25] = std::max(array[13], array[25]);
    array[13] = tmp;
    tmp = std::min(array[14], array[26]);
    array[26] = std::max(array[14], array[26]);
    array[14] = tmp;
    tmp = std::min(array[15], array[27]);
    array[27] = std::max(array[15], array[27]);
    array[15] = tmp;
    tmp = std::min(array[20], array[32]);
    array[32] = std::max(array[20], array[32]);
    array[20] = tmp;
    tmp = std::min(array[21], array[33]);
    array[33] = std::max(array[21], array[33]);
    array[21] = tmp;
    tmp = std::min(array[22], array[34]);
    array[34] = std::max(array[22], array[34]);
    array[22] = tmp;
    tmp = std::min(array[23], array[35]);
    array[35] = std::max(array[23], array[35]);
    array[23] = tmp;
    tmp = std::min(array[28], array[40]);
    array[40] = std::max(array[28], array[40]);
    array[28] = tmp;
    tmp = std::min(array[29], array[41]);
    array[41] = std::max(array[29], array[41]);
    array[29] = tmp;
    tmp = std::min(array[30], array[42]);
    array[42] = std::max(array[30], array[42]);
    array[30] = tmp;
    tmp = std::min(array[31], array[43]);
    array[43] = std::max(array[31], array[43]);
    array[31] = tmp;
    tmp = std::min(array[36], array[48]);
    array[48] = std::max(array[36], array[48]);
    array[36] = tmp;
    tmp = std::min(array[37], array[49]);
    array[49] = std::max(array[37], array[49]);
    array[37] = tmp;
    tmp = std::min(array[38], array[50]);
    array[50] = std::max(array[38], array[50]);
    array[38] = tmp;
    tmp = std::min(array[39], array[51]);
    array[51] = std::max(array[39], array[51]);
    array[39] = tmp;
    tmp = std::min(array[44], array[56]);
    array[56] = std::max(array[44], array[56]);
    array[44] = tmp;
    tmp = std::min(array[45], array[57]);
    array[57] = std::max(array[45], array[57]);
    array[45] = tmp;
    tmp = std::min(array[46], array[58]);
    array[58] = std::max(array[46], array[58]);
    array[46] = tmp;
    tmp = std::min(array[47], array[59]);
    array[59] = std::max(array[47], array[59]);
    array[47] = tmp;
    tmp = std::min(array[52], array[64]);
    array[64] = std::max(array[52], array[64]);
    array[52] = tmp;
    tmp = std::min(array[53], array[65]);
    array[65] = std::max(array[53], array[65]);
    array[53] = tmp;
    tmp = std::min(array[54], array[66]);
    array[66] = std::max(array[54], array[66]);
    array[54] = tmp;
    tmp = std::min(array[55], array[67]);
    array[67] = std::max(array[55], array[67]);
    array[55] = tmp;
    tmp = std::min(array[60], array[72]);
    array[72] = std::max(array[60], array[72]);
    array[60] = tmp;
    tmp = std::min(array[61], array[73]);
    array[73] = std::max(array[61], array[73]);
    array[61] = tmp;
    tmp = std::min(array[62], array[74]);
    array[74] = std::max(array[62], array[74]);
    array[62] = tmp;
    tmp = std::min(array[63], array[75]);
    array[75] = std::max(array[63], array[75]);
    array[63] = tmp;
    tmp = std::min(array[68], array[80]);
    array[80] = std::max(array[68], array[80]);
    array[68] = tmp;
    tmp = std::min(array[4], array[8]);
    array[8] = std::max(array[4], array[8]);
    array[4] = tmp;
    tmp = std::min(array[5], array[9]);
    array[9] = std::max(array[5], array[9]);
    array[5] = tmp;
    tmp = std::min(array[6], array[10]);
    array[10] = std::max(array[6], array[10]);
    array[6] = tmp;
    tmp = std::min(array[7], array[11]);
    array[11] = std::max(array[7], array[11]);
    array[7] = tmp;
    tmp = std::min(array[12], array[16]);
    array[16] = std::max(array[12], array[16]);
    array[12] = tmp;
    tmp = std::min(array[13], array[17]);
    array[17] = std::max(array[13], array[17]);
    array[13] = tmp;
    tmp = std::min(array[14], array[18]);
    array[18] = std::max(array[14], array[18]);
    array[14] = tmp;
    tmp = std::min(array[15], array[19]);
    array[19] = std::max(array[15], array[19]);
    array[15] = tmp;
    tmp = std::min(array[20], array[24]);
    array[24] = std::max(array[20], array[24]);
    array[20] = tmp;
    tmp = std::min(array[21], array[25]);
    array[25] = std::max(array[21], array[25]);
    array[21] = tmp;
    tmp = std::min(array[22], array[26]);
    array[26] = std::max(array[22], array[26]);
    array[22] = tmp;
    tmp = std::min(array[23], array[27]);
    array[27] = std::max(array[23], array[27]);
    array[23] = tmp;
    tmp = std::min(array[28], array[32]);
    array[32] = std::max(array[28], array[32]);
    array[28] = tmp;
    tmp = std::min(array[29], array[33]);
    array[33] = std::max(array[29], array[33]);
    array[29] = tmp;
    tmp = std::min(array[30], array[34]);
    array[34] = std::max(array[30], array[34]);
    array[30] = tmp;
    tmp = std::min(array[31], array[35]);
    array[35] = std::max(array[31], array[35]);
    array[31] = tmp;
    tmp = std::min(array[36], array[40]);
    array[40] = std::max(array[36], array[40]);
    array[36] = tmp;
    tmp = std::min(array[37], array[41]);
    array[41] = std::max(array[37], array[41]);
    array[37] = tmp;
    tmp = std::min(array[38], array[42]);
    array[42] = std::max(array[38], array[42]);
    array[38] = tmp;
    tmp = std::min(array[39], array[43]);
    array[43] = std::max(array[39], array[43]);
    array[39] = tmp;
    tmp = std::min(array[44], array[48]);
    array[48] = std::max(array[44], array[48]);
    array[44] = tmp;
    tmp = std::min(array[45], array[49]);
    array[49] = std::max(array[45], array[49]);
    array[45] = tmp;
    tmp = std::min(array[46], array[50]);
    array[50] = std::max(array[46], array[50]);
    array[46] = tmp;
    tmp = std::min(array[47], array[51]);
    array[51] = std::max(array[47], array[51]);
    array[47] = tmp;
    tmp = std::min(array[52], array[56]);
    array[56] = std::max(array[52], array[56]);
    array[52] = tmp;
    tmp = std::min(array[53], array[57]);
    array[57] = std::max(array[53], array[57]);
    array[53] = tmp;
    tmp = std::min(array[54], array[58]);
    array[58] = std::max(array[54], array[58]);
    array[54] = tmp;
    tmp = std::min(array[55], array[59]);
    array[59] = std::max(array[55], array[59]);
    array[55] = tmp;
    tmp = std::min(array[60], array[64]);
    array[64] = std::max(array[60], array[64]);
    array[60] = tmp;
    tmp = std::min(array[61], array[65]);
    array[65] = std::max(array[61], array[65]);
    array[61] = tmp;
    tmp = std::min(array[62], array[66]);
    array[66] = std::max(array[62], array[66]);
    array[62] = tmp;
    tmp = std::min(array[63], array[67]);
    array[67] = std::max(array[63], array[67]);
    array[63] = tmp;
    tmp = std::min(array[68], array[72]);
    array[72] = std::max(array[68], array[72]);
    array[68] = tmp;
    tmp = std::min(array[69], array[73]);
    array[73] = std::max(array[69], array[73]);
    array[69] = tmp;
    tmp = std::min(array[70], array[74]);
    array[74] = std::max(array[70], array[74]);
    array[70] = tmp;
    tmp = std::min(array[71], array[75]);
    array[75] = std::max(array[71], array[75]);
    array[71] = tmp;
    tmp = std::min(array[76], array[80]);
    array[80] = std::max(array[76], array[80]);
    array[76] = tmp;
    tmp = std::min(array[0], array[2]);
    array[2] = std::max(array[0], array[2]);
    array[0] = tmp;
    tmp = std::min(array[1], array[3]);
    array[3] = std::max(array[1], array[3]);
    array[1] = tmp;
    tmp = std::min(array[4], array[6]);
    array[6] = std::max(array[4], array[6]);
    array[4] = tmp;
    tmp = std::min(array[5], array[7]);
    array[7] = std::max(array[5], array[7]);
    array[5] = tmp;
    tmp = std::min(array[8], array[10]);
    array[10] = std::max(array[8], array[10]);
    array[8] = tmp;
    tmp = std::min(array[9], array[11]);
    array[11] = std::max(array[9], array[11]);
    array[9] = tmp;
    tmp = std::min(array[12], array[14]);
    array[14] = std::max(array[12], array[14]);
    array[12] = tmp;
    tmp = std::min(array[13], array[15]);
    array[15] = std::max(array[13], array[15]);
    array[13] = tmp;
    tmp = std::min(array[16], array[18]);
    array[18] = std::max(array[16], array[18]);
    array[16] = tmp;
    tmp = std::min(array[17], array[19]);
    array[19] = std::max(array[17], array[19]);
    array[17] = tmp;
    tmp = std::min(array[20], array[22]);
    array[22] = std::max(array[20], array[22]);
    array[20] = tmp;
    tmp = std::min(array[21], array[23]);
    array[23] = std::max(array[21], array[23]);
    array[21] = tmp;
    tmp = std::min(array[24], array[26]);
    array[26] = std::max(array[24], array[26]);
    array[24] = tmp;
    tmp = std::min(array[25], array[27]);
    array[27] = std::max(array[25], array[27]);
    array[25] = tmp;
    tmp = std::min(array[28], array[30]);
    array[30] = std::max(array[28], array[30]);
    array[28] = tmp;
    tmp = std::min(array[29], array[31]);
    array[31] = std::max(array[29], array[31]);
    array[29] = tmp;
    tmp = std::min(array[32], array[34]);
    array[34] = std::max(array[32], array[34]);
    array[32] = tmp;
    tmp = std::min(array[33], array[35]);
    array[35] = std::max(array[33], array[35]);
    array[33] = tmp;
    tmp = std::min(array[36], array[38]);
    array[38] = std::max(array[36], array[38]);
    array[36] = tmp;
    tmp = std::min(array[37], array[39]);
    array[39] = std::max(array[37], array[39]);
    array[37] = tmp;
    tmp = std::min(array[40], array[42]);
    array[42] = std::max(array[40], array[42]);
    array[40] = tmp;
    tmp = std::min(array[41], array[43]);
    array[43] = std::max(array[41], array[43]);
    array[41] = tmp;
    tmp = std::min(array[44], array[46]);
    array[46] = std::max(array[44], array[46]);
    array[44] = tmp;
    tmp = std::min(array[45], array[47]);
    array[47] = std::max(array[45], array[47]);
    array[45] = tmp;
    tmp = std::min(array[48], array[50]);
    array[50] = std::max(array[48], array[50]);
    array[48] = tmp;
    tmp = std::min(array[49], array[51]);
    array[51] = std::max(array[49], array[51]);
    array[49] = tmp;
    tmp = std::min(array[52], array[54]);
    array[54] = std::max(array[52], array[54]);
    array[52] = tmp;
    tmp = std::min(array[53], array[55]);
    array[55] = std::max(array[53], array[55]);
    array[53] = tmp;
    tmp = std::min(array[56], array[58]);
    array[58] = std::max(array[56], array[58]);
    array[56] = tmp;
    tmp = std::min(array[57], array[59]);
    array[59] = std::max(array[57], array[59]);
    array[57] = tmp;
    tmp = std::min(array[60], array[62]);
    array[62] = std::max(array[60], array[62]);
    array[60] = tmp;
    tmp = std::min(array[61], array[63]);
    array[63] = std::max(array[61], array[63]);
    array[61] = tmp;
    tmp = std::min(array[64], array[66]);
    array[66] = std::max(array[64], array[66]);
    array[64] = tmp;
    tmp = std::min(array[65], array[67]);
    array[67] = std::max(array[65], array[67]);
    array[65] = tmp;
    tmp = std::min(array[68], array[70]);
    array[70] = std::max(array[68], array[70]);
    array[68] = tmp;
    tmp = std::min(array[69], array[71]);
    array[71] = std::max(array[69], array[71]);
    array[69] = tmp;
    tmp = std::min(array[72], array[74]);
    array[74] = std::max(array[72], array[74]);
    array[72] = tmp;
    tmp = std::min(array[73], array[75]);
    array[75] = std::max(array[73], array[75]);
    array[73] = tmp;
    tmp = std::min(array[76], array[78]);
    array[78] = std::max(array[76], array[78]);
    array[76] = tmp;
    tmp = std::min(array[77], array[79]);
    array[79] = std::max(array[77], array[79]);
    array[77] = tmp;
    tmp = std::min(array[2], array[64]);
    array[64] = std::max(array[2], array[64]);
    array[2] = tmp;
    tmp = std::min(array[3], array[65]);
    array[65] = std::max(array[3], array[65]);
    array[3] = tmp;
    tmp = std::min(array[6], array[68]);
    array[68] = std::max(array[6], array[68]);
    array[6] = tmp;
    tmp = std::min(array[7], array[69]);
    array[69] = std::max(array[7], array[69]);
    array[7] = tmp;
    tmp = std::min(array[10], array[72]);
    array[72] = std::max(array[10], array[72]);
    array[10] = tmp;
    tmp = std::min(array[11], array[73]);
    array[73] = std::max(array[11], array[73]);
    array[11] = tmp;
    tmp = std::min(array[14], array[76]);
    array[76] = std::max(array[14], array[76]);
    array[14] = tmp;
    tmp = std::min(array[15], array[77]);
    array[77] = std::max(array[15], array[77]);
    array[15] = tmp;
    tmp = std::min(array[18], array[80]);
    array[80] = std::max(array[18], array[80]);
    array[18] = tmp;
    tmp = std::min(array[2], array[32]);
    array[32] = std::max(array[2], array[32]);
    array[2] = tmp;
    tmp = std::min(array[3], array[33]);
    array[33] = std::max(array[3], array[33]);
    array[3] = tmp;
    tmp = std::min(array[6], array[36]);
    array[36] = std::max(array[6], array[36]);
    array[6] = tmp;
    tmp = std::min(array[7], array[37]);
    array[37] = std::max(array[7], array[37]);
    array[7] = tmp;
    tmp = std::min(array[10], array[40]);
    array[40] = std::max(array[10], array[40]);
    array[10] = tmp;
    tmp = std::min(array[11], array[41]);
    array[41] = std::max(array[11], array[41]);
    array[11] = tmp;
    tmp = std::min(array[14], array[44]);
    array[44] = std::max(array[14], array[44]);
    array[14] = tmp;
    tmp = std::min(array[15], array[45]);
    array[45] = std::max(array[15], array[45]);
    array[15] = tmp;
    tmp = std::min(array[18], array[48]);
    array[48] = std::max(array[18], array[48]);
    array[18] = tmp;
    tmp = std::min(array[19], array[49]);
    array[49] = std::max(array[19], array[49]);
    array[19] = tmp;
    tmp = std::min(array[22], array[52]);
    array[52] = std::max(array[22], array[52]);
    array[22] = tmp;
    tmp = std::min(array[23], array[53]);
    array[53] = std::max(array[23], array[53]);
    array[23] = tmp;
    tmp = std::min(array[26], array[56]);
    array[56] = std::max(array[26], array[56]);
    array[26] = tmp;
    tmp = std::min(array[27], array[57]);
    array[57] = std::max(array[27], array[57]);
    array[27] = tmp;
    tmp = std::min(array[30], array[60]);
    array[60] = std::max(array[30], array[60]);
    array[30] = tmp;
    tmp = std::min(array[31], array[61]);
    array[61] = std::max(array[31], array[61]);
    array[31] = tmp;
    tmp = std::min(array[34], array[64]);
    array[64] = std::max(array[34], array[64]);
    array[34] = tmp;
    tmp = std::min(array[35], array[65]);
    array[65] = std::max(array[35], array[65]);
    array[35] = tmp;
    tmp = std::min(array[38], array[68]);
    array[68] = std::max(array[38], array[68]);
    array[38] = tmp;
    tmp = std::min(array[39], array[69]);
    array[69] = std::max(array[39], array[69]);
    array[39] = tmp;
    tmp = std::min(array[42], array[72]);
    array[72] = std::max(array[42], array[72]);
    array[42] = tmp;
    tmp = std::min(array[43], array[73]);
    array[73] = std::max(array[43], array[73]);
    array[43] = tmp;
    tmp = std::min(array[46], array[76]);
    array[76] = std::max(array[46], array[76]);
    array[46] = tmp;
    tmp = std::min(array[47], array[77]);
    array[77] = std::max(array[47], array[77]);
    array[47] = tmp;
    tmp = std::min(array[50], array[80]);
    array[80] = std::max(array[50], array[80]);
    array[50] = tmp;
    tmp = std::min(array[2], array[16]);
    array[16] = std::max(array[2], array[16]);
    array[2] = tmp;
    tmp = std::min(array[3], array[17]);
    array[17] = std::max(array[3], array[17]);
    array[3] = tmp;
    tmp = std::min(array[6], array[20]);
    array[20] = std::max(array[6], array[20]);
    array[6] = tmp;
    tmp = std::min(array[7], array[21]);
    array[21] = std::max(array[7], array[21]);
    array[7] = tmp;
    tmp = std::min(array[10], array[24]);
    array[24] = std::max(array[10], array[24]);
    array[10] = tmp;
    tmp = std::min(array[11], array[25]);
    array[25] = std::max(array[11], array[25]);
    array[11] = tmp;
    tmp = std::min(array[14], array[28]);
    array[28] = std::max(array[14], array[28]);
    array[14] = tmp;
    tmp = std::min(array[15], array[29]);
    array[29] = std::max(array[15], array[29]);
    array[15] = tmp;
    tmp = std::min(array[18], array[32]);
    array[32] = std::max(array[18], array[32]);
    array[18] = tmp;
    tmp = std::min(array[19], array[33]);
    array[33] = std::max(array[19], array[33]);
    array[19] = tmp;
    tmp = std::min(array[22], array[36]);
    array[36] = std::max(array[22], array[36]);
    array[22] = tmp;
    tmp = std::min(array[23], array[37]);
    array[37] = std::max(array[23], array[37]);
    array[23] = tmp;
    tmp = std::min(array[26], array[40]);
    array[40] = std::max(array[26], array[40]);
    array[26] = tmp;
    tmp = std::min(array[27], array[41]);
    array[41] = std::max(array[27], array[41]);
    array[27] = tmp;
    tmp = std::min(array[30], array[44]);
    array[44] = std::max(array[30], array[44]);
    array[30] = tmp;
    tmp = std::min(array[31], array[45]);
    array[45] = std::max(array[31], array[45]);
    array[31] = tmp;
    tmp = std::min(array[34], array[48]);
    array[48] = std::max(array[34], array[48]);
    array[34] = tmp;
    tmp = std::min(array[35], array[49]);
    array[49] = std::max(array[35], array[49]);
    array[35] = tmp;
    tmp = std::min(array[38], array[52]);
    array[52] = std::max(array[38], array[52]);
    array[38] = tmp;
    tmp = std::min(array[39], array[53]);
    array[53] = std::max(array[39], array[53]);
    array[39] = tmp;
    tmp = std::min(array[42], array[56]);
    array[56] = std::max(array[42], array[56]);
    array[42] = tmp;
    tmp = std::min(array[43], array[57]);
    array[57] = std::max(array[43], array[57]);
    array[43] = tmp;
    tmp = std::min(array[46], array[60]);
    array[60] = std::max(array[46], array[60]);
    array[46] = tmp;
    tmp = std::min(array[47], array[61]);
    array[61] = std::max(array[47], array[61]);
    array[47] = tmp;
    tmp = std::min(array[50], array[64]);
    array[64] = std::max(array[50], array[64]);
    array[50] = tmp;
    tmp = std::min(array[51], array[65]);
    array[65] = std::max(array[51], array[65]);
    array[51] = tmp;
    tmp = std::min(array[54], array[68]);
    array[68] = std::max(array[54], array[68]);
    array[54] = tmp;
    tmp = std::min(array[55], array[69]);
    array[69] = std::max(array[55], array[69]);
    array[55] = tmp;
    tmp = std::min(array[58], array[72]);
    array[72] = std::max(array[58], array[72]);
    array[58] = tmp;
    tmp = std::min(array[59], array[73]);
    array[73] = std::max(array[59], array[73]);
    array[59] = tmp;
    tmp = std::min(array[62], array[76]);
    array[76] = std::max(array[62], array[76]);
    array[62] = tmp;
    tmp = std::min(array[63], array[77]);
    array[77] = std::max(array[63], array[77]);
    array[63] = tmp;
    tmp = std::min(array[66], array[80]);
    array[80] = std::max(array[66], array[80]);
    array[66] = tmp;
    tmp = std::min(array[2], array[8]);
    array[8] = std::max(array[2], array[8]);
    array[2] = tmp;
    tmp = std::min(array[3], array[9]);
    array[9] = std::max(array[3], array[9]);
    array[3] = tmp;
    tmp = std::min(array[6], array[12]);
    array[12] = std::max(array[6], array[12]);
    array[6] = tmp;
    tmp = std::min(array[7], array[13]);
    array[13] = std::max(array[7], array[13]);
    array[7] = tmp;
    tmp = std::min(array[10], array[16]);
    array[16] = std::max(array[10], array[16]);
    array[10] = tmp;
    tmp = std::min(array[11], array[17]);
    array[17] = std::max(array[11], array[17]);
    array[11] = tmp;
    tmp = std::min(array[14], array[20]);
    array[20] = std::max(array[14], array[20]);
    array[14] = tmp;
    tmp = std::min(array[15], array[21]);
    array[21] = std::max(array[15], array[21]);
    array[15] = tmp;
    tmp = std::min(array[18], array[24]);
    array[24] = std::max(array[18], array[24]);
    array[18] = tmp;
    tmp = std::min(array[19], array[25]);
    array[25] = std::max(array[19], array[25]);
    array[19] = tmp;
    tmp = std::min(array[22], array[28]);
    array[28] = std::max(array[22], array[28]);
    array[22] = tmp;
    tmp = std::min(array[23], array[29]);
    array[29] = std::max(array[23], array[29]);
    array[23] = tmp;
    tmp = std::min(array[26], array[32]);
    array[32] = std::max(array[26], array[32]);
    array[26] = tmp;
    tmp = std::min(array[27], array[33]);
    array[33] = std::max(array[27], array[33]);
    array[27] = tmp;
    tmp = std::min(array[30], array[36]);
    array[36] = std::max(array[30], array[36]);
    array[30] = tmp;
    tmp = std::min(array[31], array[37]);
    array[37] = std::max(array[31], array[37]);
    array[31] = tmp;
    tmp = std::min(array[34], array[40]);
    array[40] = std::max(array[34], array[40]);
    array[34] = tmp;
    tmp = std::min(array[35], array[41]);
    array[41] = std::max(array[35], array[41]);
    array[35] = tmp;
    tmp = std::min(array[38], array[44]);
    array[44] = std::max(array[38], array[44]);
    array[38] = tmp;
    tmp = std::min(array[39], array[45]);
    array[45] = std::max(array[39], array[45]);
    array[39] = tmp;
    tmp = std::min(array[42], array[48]);
    array[48] = std::max(array[42], array[48]);
    array[42] = tmp;
    tmp = std::min(array[43], array[49]);
    array[49] = std::max(array[43], array[49]);
    array[43] = tmp;
    tmp = std::min(array[46], array[52]);
    array[52] = std::max(array[46], array[52]);
    array[46] = tmp;
    tmp = std::min(array[47], array[53]);
    array[53] = std::max(array[47], array[53]);
    array[47] = tmp;
    tmp = std::min(array[50], array[56]);
    array[56] = std::max(array[50], array[56]);
    array[50] = tmp;
    tmp = std::min(array[51], array[57]);
    array[57] = std::max(array[51], array[57]);
    array[51] = tmp;
    tmp = std::min(array[54], array[60]);
    array[60] = std::max(array[54], array[60]);
    array[54] = tmp;
    tmp = std::min(array[55], array[61]);
    array[61] = std::max(array[55], array[61]);
    array[55] = tmp;
    tmp = std::min(array[58], array[64]);
    array[64] = std::max(array[58], array[64]);
    array[58] = tmp;
    tmp = std::min(array[59], array[65]);
    array[65] = std::max(array[59], array[65]);
    array[59] = tmp;
    tmp = std::min(array[62], array[68]);
    array[68] = std::max(array[62], array[68]);
    array[62] = tmp;
    tmp = std::min(array[63], array[69]);
    array[69] = std::max(array[63], array[69]);
    array[63] = tmp;
    tmp = std::min(array[66], array[72]);
    array[72] = std::max(array[66], array[72]);
    array[66] = tmp;
    tmp = std::min(array[67], array[73]);
    array[73] = std::max(array[67], array[73]);
    array[67] = tmp;
    tmp = std::min(array[70], array[76]);
    array[76] = std::max(array[70], array[76]);
    array[70] = tmp;
    tmp = std::min(array[71], array[77]);
    array[77] = std::max(array[71], array[77]);
    array[71] = tmp;
    tmp = std::min(array[74], array[80]);
    array[80] = std::max(array[74], array[80]);
    array[74] = tmp;
    tmp = std::min(array[2], array[4]);
    array[4] = std::max(array[2], array[4]);
    array[2] = tmp;
    tmp = std::min(array[3], array[5]);
    array[5] = std::max(array[3], array[5]);
    array[3] = tmp;
    tmp = std::min(array[6], array[8]);
    array[8] = std::max(array[6], array[8]);
    array[6] = tmp;
    tmp = std::min(array[7], array[9]);
    array[9] = std::max(array[7], array[9]);
    array[7] = tmp;
    tmp = std::min(array[10], array[12]);
    array[12] = std::max(array[10], array[12]);
    array[10] = tmp;
    tmp = std::min(array[11], array[13]);
    array[13] = std::max(array[11], array[13]);
    array[11] = tmp;
    tmp = std::min(array[14], array[16]);
    array[16] = std::max(array[14], array[16]);
    array[14] = tmp;
    tmp = std::min(array[15], array[17]);
    array[17] = std::max(array[15], array[17]);
    array[15] = tmp;
    tmp = std::min(array[18], array[20]);
    array[20] = std::max(array[18], array[20]);
    array[18] = tmp;
    tmp = std::min(array[19], array[21]);
    array[21] = std::max(array[19], array[21]);
    array[19] = tmp;
    tmp = std::min(array[22], array[24]);
    array[24] = std::max(array[22], array[24]);
    array[22] = tmp;
    tmp = std::min(array[23], array[25]);
    array[25] = std::max(array[23], array[25]);
    array[23] = tmp;
    tmp = std::min(array[26], array[28]);
    array[28] = std::max(array[26], array[28]);
    array[26] = tmp;
    tmp = std::min(array[27], array[29]);
    array[29] = std::max(array[27], array[29]);
    array[27] = tmp;
    tmp = std::min(array[30], array[32]);
    array[32] = std::max(array[30], array[32]);
    array[30] = tmp;
    tmp = std::min(array[31], array[33]);
    array[33] = std::max(array[31], array[33]);
    array[31] = tmp;
    tmp = std::min(array[34], array[36]);
    array[36] = std::max(array[34], array[36]);
    array[34] = tmp;
    tmp = std::min(array[35], array[37]);
    array[37] = std::max(array[35], array[37]);
    array[35] = tmp;
    tmp = std::min(array[38], array[40]);
    array[40] = std::max(array[38], array[40]);
    array[38] = tmp;
    tmp = std::min(array[39], array[41]);
    array[41] = std::max(array[39], array[41]);
    array[39] = tmp;
    tmp = std::min(array[42], array[44]);
    array[44] = std::max(array[42], array[44]);
    array[42] = tmp;
    tmp = std::min(array[43], array[45]);
    array[45] = std::max(array[43], array[45]);
    array[43] = tmp;
    tmp = std::min(array[46], array[48]);
    array[48] = std::max(array[46], array[48]);
    array[46] = tmp;
    tmp = std::min(array[47], array[49]);
    array[49] = std::max(array[47], array[49]);
    array[47] = tmp;
    tmp = std::min(array[50], array[52]);
    array[52] = std::max(array[50], array[52]);
    array[50] = tmp;
    tmp = std::min(array[51], array[53]);
    array[53] = std::max(array[51], array[53]);
    array[51] = tmp;
    tmp = std::min(array[54], array[56]);
    array[56] = std::max(array[54], array[56]);
    array[54] = tmp;
    tmp = std::min(array[55], array[57]);
    array[57] = std::max(array[55], array[57]);
    array[55] = tmp;
    tmp = std::min(array[58], array[60]);
    array[60] = std::max(array[58], array[60]);
    array[58] = tmp;
    tmp = std::min(array[59], array[61]);
    array[61] = std::max(array[59], array[61]);
    array[59] = tmp;
    tmp = std::min(array[62], array[64]);
    array[64] = std::max(array[62], array[64]);
    array[62] = tmp;
    tmp = std::min(array[63], array[65]);
    array[65] = std::max(array[63], array[65]);
    array[63] = tmp;
    tmp = std::min(array[66], array[68]);
    array[68] = std::max(array[66], array[68]);
    array[66] = tmp;
    tmp = std::min(array[67], array[69]);
    array[69] = std::max(array[67], array[69]);
    array[67] = tmp;
    tmp = std::min(array[70], array[72]);
    array[72] = std::max(array[70], array[72]);
    array[70] = tmp;
    tmp = std::min(array[71], array[73]);
    array[73] = std::max(array[71], array[73]);
    array[71] = tmp;
    tmp = std::min(array[74], array[76]);
    array[76] = std::max(array[74], array[76]);
    array[74] = tmp;
    tmp = std::min(array[75], array[77]);
    array[77] = std::max(array[75], array[77]);
    array[75] = tmp;
    tmp = std::min(array[78], array[80]);
    array[80] = std::max(array[78], array[80]);
    array[78] = tmp;
    array[1] = std::max(array[0], array[1]);
    array[3] = std::max(array[2], array[3]);
    array[5] = std::max(array[4], array[5]);
    array[7] = std::max(array[6], array[7]);
    array[9] = std::max(array[8], array[9]);
    array[11] = std::max(array[10], array[11]);
    array[13] = std::max(array[12], array[13]);
    array[15] = std::max(array[14], array[15]);
    array[17] = std::max(array[16], array[17]);
    array[19] = std::max(array[18], array[19]);
    array[21] = std::max(array[20], array[21]);
    array[23] = std::max(array[22], array[23]);
    array[25] = std::max(array[24], array[25]);
    array[27] = std::max(array[26], array[27]);
    array[29] = std::max(array[28], array[29]);
    array[31] = std::max(array[30], array[31]);
    array[33] = std::max(array[32], array[33]);
    array[35] = std::max(array[34], array[35]);
    array[37] = std::max(array[36], array[37]);
    array[39] = std::max(array[38], array[39]);
    array[40] = std::min(array[40], array[41]);
    array[42] = std::min(array[42], array[43]);
    array[44] = std::min(array[44], array[45]);
    array[46] = std::min(array[46], array[47]);
    array[48] = std::min(array[48], array[49]);
    array[50] = std::min(array[50], array[51]);
    array[52] = std::min(array[52], array[53]);
    array[54] = std::min(array[54], array[55]);
    array[56] = std::min(array[56], array[57]);
    array[58] = std::min(array[58], array[59]);
    array[60] = std::min(array[60], array[61]);
    array[62] = std::min(array[62], array[63]);
    array[64] = std::min(array[64], array[65]);
    array[66] = std::min(array[66], array[67]);
    array[68] = std::min(array[68], array[69]);
    array[70] = std::min(array[70], array[71]);
    array[72] = std::min(array[72], array[73]);
    array[74] = std::min(array[74], array[75]);
    array[76] = std::min(array[76], array[77]);
    array[78] = std::min(array[78], array[79]);
    array[64] = std::max(array[1], array[64]);
    array[66] = std::max(array[3], array[66]);
    array[68] = std::max(array[5], array[68]);
    array[70] = std::max(array[7], array[70]);
    array[9] = std::min(array[9], array[72]);
    array[11] = std::min(array[11], array[74]);
    array[13] = std::min(array[13], array[76]);
    array[15] = std::min(array[15], array[78]);
    array[17] = std::min(array[17], array[80]);
    array[40] = std::max(array[9], array[40]);
    array[42] = std::max(array[11], array[42]);
    array[44] = std::max(array[13], array[44]);
    array[46] = std::max(array[15], array[46]);
    array[48] = std::max(array[17], array[48]);
    array[50] = std::max(array[19], array[50]);
    array[52] = std::max(array[21], array[52]);
    array[54] = std::max(array[23], array[54]);
    array[25] = std::min(array[25], array[56]);
    array[27] = std::min(array[27], array[58]);
    array[29] = std::min(array[29], array[60]);
    array[31] = std::min(array[31], array[62]);
    array[33] = std::min(array[33], array[64]);
    array[35] = std::min(array[35], array[66]);
    array[37] = std::min(array[37], array[68]);
    array[39] = std::min(array[39], array[70]);
    array[40] = std::max(array[25], array[40]);
    array[42] = std::max(array[27], array[42]);
    array[44] = std::max(array[29], array[44]);
    array[46] = std::max(array[31], array[46]);
    array[33] = std::min(array[33], array[48]);
    array[35] = std::min(array[35], array[50]);
    array[37] = std::min(array[37], array[52]);
    array[39] = std::min(array[39], array[54]);
    array[40] = std::max(array[33], array[40]);
    array[42] = std::max(array[35], array[42]);
    array[37] = std::min(array[37], array[44]);
    array[39] = std::min(array[39], array[46]);
    array[40] = std::max(array[37], array[40]);
    array[39] = std::min(array[39], array[42]);
    return std::max(array[39], array[40]);
}

#ifdef __SSE2__
template<>
inline vfloat median(std::array<vfloat, 81> array)
{
    vfloat tmp = vminf(array[0], array[64]);
    array[64] = vmaxf(array[0], array[64]);
    array[0] = tmp;
    tmp = vminf(array[1], array[65]);
    array[65] = vmaxf(array[1], array[65]);
    array[1] = tmp;
    tmp = vminf(array[2], array[66]);
    array[66] = vmaxf(array[2], array[66]);
    array[2] = tmp;
    tmp = vminf(array[3], array[67]);
    array[67] = vmaxf(array[3], array[67]);
    array[3] = tmp;
    tmp = vminf(array[4], array[68]);
    array[68] = vmaxf(array[4], array[68]);
    array[4] = tmp;
    tmp = vminf(array[5], array[69]);
    array[69] = vmaxf(array[5], array[69]);
    array[5] = tmp;
    tmp = vminf(array[6], array[70]);
    array[70] = vmaxf(array[6], array[70]);
    array[6] = tmp;
    tmp = vminf(array[7], array[71]);
    array[71] = vmaxf(array[7], array[71]);
    array[7] = tmp;
    tmp = vminf(array[8], array[72]);
    array[72] = vmaxf(array[8], array[72]);
    array[8] = tmp;
    tmp = vminf(array[9], array[73]);
    array[73] = vmaxf(array[9], array[73]);
    array[9] = tmp;
    tmp = vminf(array[10], array[74]);
    array[74] = vmaxf(array[10], array[74]);
    array[10] = tmp;
    tmp = vminf(array[11], array[75]);
    array[75] = vmaxf(array[11], array[75]);
    array[11] = tmp;
    tmp = vminf(array[12], array[76]);
    array[76] = vmaxf(array[12], array[76]);
    array[12] = tmp;
    tmp = vminf(array[13], array[77]);
    array[77] = vmaxf(array[13], array[77]);
    array[13] = tmp;
    tmp = vminf(array[14], array[78]);
    array[78] = vmaxf(array[14], array[78]);
    array[14] = tmp;
    tmp = vminf(array[15], array[79]);
    array[79] = vmaxf(array[15], array[79]);
    array[15] = tmp;
    tmp = vminf(array[16], array[80]);
    array[80] = vmaxf(array[16], array[80]);
    array[16] = tmp;
    tmp = vminf(array[0], array[32]);
    array[32] = vmaxf(array[0], array[32]);
    array[0] = tmp;
    tmp = vminf(array[1], array[33]);
    array[33] = vmaxf(array[1], array[33]);
    array[1] = tmp;
    tmp = vminf(array[2], array[34]);
    array[34] = vmaxf(array[2], array[34]);
    array[2] = tmp;
    tmp = vminf(array[3], array[35]);
    array[35] = vmaxf(array[3], array[35]);
    array[3] = tmp;
    tmp = vminf(array[4], array[36]);
    array[36] = vmaxf(array[4], array[36]);
    array[4] = tmp;
    tmp = vminf(array[5], array[37]);
    array[37] = vmaxf(array[5], array[37]);
    array[5] = tmp;
    tmp = vminf(array[6], array[38]);
    array[38] = vmaxf(array[6], array[38]);
    array[6] = tmp;
    tmp = vminf(array[7], array[39]);
    array[39] = vmaxf(array[7], array[39]);
    array[7] = tmp;
    tmp = vminf(array[8], array[40]);
    array[40] = vmaxf(array[8], array[40]);
    array[8] = tmp;
    tmp = vminf(array[9], array[41]);
    array[41] = vmaxf(array[9], array[41]);
    array[9] = tmp;
    tmp = vminf(array[10], array[42]);
    array[42] = vmaxf(array[10], array[42]);
    array[10] = tmp;
    tmp = vminf(array[11], array[43]);
    array[43] = vmaxf(array[11], array[43]);
    array[11] = tmp;
    tmp = vminf(array[12], array[44]);
    array[44] = vmaxf(array[12], array[44]);
    array[12] = tmp;
    tmp = vminf(array[13], array[45]);
    array[45] = vmaxf(array[13], array[45]);
    array[13] = tmp;
    tmp = vminf(array[14], array[46]);
    array[46] = vmaxf(array[14], array[46]);
    array[14] = tmp;
    tmp = vminf(array[15], array[47]);
    array[47] = vmaxf(array[15], array[47]);
    array[15] = tmp;
    tmp = vminf(array[16], array[48]);
    array[48] = vmaxf(array[16], array[48]);
    array[16] = tmp;
    tmp = vminf(array[17], array[49]);
    array[49] = vmaxf(array[17], array[49]);
    array[17] = tmp;
    tmp = vminf(array[18], array[50]);
    array[50] = vmaxf(array[18], array[50]);
    array[18] = tmp;
    tmp = vminf(array[19], array[51]);
    array[51] = vmaxf(array[19], array[51]);
    array[19] = tmp;
    tmp = vminf(array[20], array[52]);
    array[52] = vmaxf(array[20], array[52]);
    array[20] = tmp;
    tmp = vminf(array[21], array[53]);
    array[53] = vmaxf(array[21], array[53]);
    array[21] = tmp;
    tmp = vminf(array[22], array[54]);
    array[54] = vmaxf(array[22], array[54]);
    array[22] = tmp;
    tmp = vminf(array[23], array[55]);
    array[55] = vmaxf(array[23], array[55]);
    array[23] = tmp;
    tmp = vminf(array[24], array[56]);
    array[56] = vmaxf(array[24], array[56]);
    array[24] = tmp;
    tmp = vminf(array[25], array[57]);
    array[57] = vmaxf(array[25], array[57]);
    array[25] = tmp;
    tmp = vminf(array[26], array[58]);
    array[58] = vmaxf(array[26], array[58]);
    array[26] = tmp;
    tmp = vminf(array[27], array[59]);
    array[59] = vmaxf(array[27], array[59]);
    array[27] = tmp;
    tmp = vminf(array[28], array[60]);
    array[60] = vmaxf(array[28], array[60]);
    array[28] = tmp;
    tmp = vminf(array[29], array[61]);
    array[61] = vmaxf(array[29], array[61]);
    array[29] = tmp;
    tmp = vminf(array[30], array[62]);
    array[62] = vmaxf(array[30], array[62]);
    array[30] = tmp;
    tmp = vminf(array[31], array[63]);
    array[63] = vmaxf(array[31], array[63]);
    array[31] = tmp;
    tmp = vminf(array[32], array[64]);
    array[64] = vmaxf(array[32], array[64]);
    array[32] = tmp;
    tmp = vminf(array[33], array[65]);
    array[65] = vmaxf(array[33], array[65]);
    array[33] = tmp;
    tmp = vminf(array[34], array[66]);
    array[66] = vmaxf(array[34], array[66]);
    array[34] = tmp;
    tmp = vminf(array[35], array[67]);
    array[67] = vmaxf(array[35], array[67]);
    array[35] = tmp;
    tmp = vminf(array[36], array[68]);
    array[68] = vmaxf(array[36], array[68]);
    array[36] = tmp;
    tmp = vminf(array[37], array[69]);
    array[69] = vmaxf(array[37], array[69]);
    array[37] = tmp;
    tmp = vminf(array[38], array[70]);
    array[70] = vmaxf(array[38], array[70]);
    array[38] = tmp;
    tmp = vminf(array[39], array[71]);
    array[71] = vmaxf(array[39], array[71]);
    array[39] = tmp;
    tmp = vminf(array[40], array[72]);
    array[72] = vmaxf(array[40], array[72]);
    array[40] = tmp;
    tmp = vminf(array[41], array[73]);
    array[73] = vmaxf(array[41], array[73]);
    array[41] = tmp;
    tmp = vminf(array[42], array[74]);
    array[74] = vmaxf(array[42], array[74]);
    array[42] = tmp;
    tmp = vminf(array[43], array[75]);
    array[75] = vmaxf(array[43], array[75]);
    array[43] = tmp;
    tmp = vminf(array[44], array[76]);
    array[76] = vmaxf(array[44], array[76]);
    array[44] = tmp;
    tmp = vminf(array[45], array[77]);
    array[77] = vmaxf(array[45], array[77]);
    array[45] = tmp;
    tmp = vminf(array[46], array[78]);
    array[78] = vmaxf(array[46], array[78]);
    array[46] = tmp;
    tmp = vminf(array[47], array[79]);
    array[79] = vmaxf(array[47], array[79]);
    array[47] = tmp;
    tmp = vminf(array[48], array[80]);
    array[80] = vmaxf(array[48], array[80]);
    array[48] = tmp;
    tmp = vminf(array[0], array[16]);
    array[16] = vmaxf(array[0], array[16]);
    array[0] = tmp;
    tmp = vminf(array[1], array[17]);
    array[17] = vmaxf(array[1], array[17]);
    array[1] = tmp;
    tmp = vminf(array[2], array[18]);
    array[18] = vmaxf(array[2], array[18]);
    array[2] = tmp;
    tmp = vminf(array[3], array[19]);
    array[19] = vmaxf(array[3], array[19]);
    array[3] = tmp;
    tmp = vminf(array[4], array[20]);
    array[20] = vmaxf(array[4], array[20]);
    array[4] = tmp;
    tmp = vminf(array[5], array[21]);
    array[21] = vmaxf(array[5], array[21]);
    array[5] = tmp;
    tmp = vminf(array[6], array[22]);
    array[22] = vmaxf(array[6], array[22]);
    array[6] = tmp;
    tmp = vminf(array[7], array[23]);
    array[23] = vmaxf(array[7], array[23]);
    array[7] = tmp;
    tmp = vminf(array[8], array[24]);
    array[24] = vmaxf(array[8], array[24]);
    array[8] = tmp;
    tmp = vminf(array[9], array[25]);
    array[25] = vmaxf(array[9], array[25]);
    array[9] = tmp;
    tmp = vminf(array[10], array[26]);
    array[26] = vmaxf(array[10], array[26]);
    array[10] = tmp;
    tmp = vminf(array[11], array[27]);
    array[27] = vmaxf(array[11], array[27]);
    array[11] = tmp;
    tmp = vminf(array[12], array[28]);
    array[28] = vmaxf(array[12], array[28]);
    array[12] = tmp;
    tmp = vminf(array[13], array[29]);
    array[29] = vmaxf(array[13], array[29]);
    array[13] = tmp;
    tmp = vminf(array[14], array[30]);
    array[30] = vmaxf(array[14], array[30]);
    array[14] = tmp;
    tmp = vminf(array[15], array[31]);
    array[31] = vmaxf(array[15], array[31]);
    array[15] = tmp;
    tmp = vminf(array[32], array[48]);
    array[48] = vmaxf(array[32], array[48]);
    array[32] = tmp;
    tmp = vminf(array[33], array[49]);
    array[49] = vmaxf(array[33], array[49]);
    array[33] = tmp;
    tmp = vminf(array[34], array[50]);
    array[50] = vmaxf(array[34], array[50]);
    array[34] = tmp;
    tmp = vminf(array[35], array[51]);
    array[51] = vmaxf(array[35], array[51]);
    array[35] = tmp;
    tmp = vminf(array[36], array[52]);
    array[52] = vmaxf(array[36], array[52]);
    array[36] = tmp;
    tmp = vminf(array[37], array[53]);
    array[53] = vmaxf(array[37], array[53]);
    array[37] = tmp;
    tmp = vminf(array[38], array[54]);
    array[54] = vmaxf(array[38], array[54]);
    array[38] = tmp;
    tmp = vminf(array[39], array[55]);
    array[55] = vmaxf(array[39], array[55]);
    array[39] = tmp;
    tmp = vminf(array[40], array[56]);
    array[56] = vmaxf(array[40], array[56]);
    array[40] = tmp;
    tmp = vminf(array[41], array[57]);
    array[57] = vmaxf(array[41], array[57]);
    array[41] = tmp;
    tmp = vminf(array[42], array[58]);
    array[58] = vmaxf(array[42], array[58]);
    array[42] = tmp;
    tmp = vminf(array[43], array[59]);
    array[59] = vmaxf(array[43], array[59]);
    array[43] = tmp;
    tmp = vminf(array[44], array[60]);
    array[60] = vmaxf(array[44], array[60]);
    array[44] = tmp;
    tmp = vminf(array[45], array[61]);
    array[61] = vmaxf(array[45], array[61]);
    array[45] = tmp;
    tmp = vminf(array[46], array[62]);
    array[62] = vmaxf(array[46], array[62]);
    array[46] = tmp;
    tmp = vminf(array[47], array[63]);
    array[63] = vmaxf(array[47], array[63]);
    array[47] = tmp;
    tmp = vminf(array[64], array[80]);
    array[80] = vmaxf(array[64], array[80]);
    array[64] = tmp;
    tmp = vminf(array[16], array[64]);
    array[64] = vmaxf(array[16], array[64]);
    array[16] = tmp;
    tmp = vminf(array[17], array[65]);
    array[65] = vmaxf(array[17], array[65]);
    array[17] = tmp;
    tmp = vminf(array[18], array[66]);
    array[66] = vmaxf(array[18], array[66]);
    array[18] = tmp;
    tmp = vminf(array[19], array[67]);
    array[67] = vmaxf(array[19], array[67]);
    array[19] = tmp;
    tmp = vminf(array[20], array[68]);
    array[68] = vmaxf(array[20], array[68]);
    array[20] = tmp;
    tmp = vminf(array[21], array[69]);
    array[69] = vmaxf(array[21], array[69]);
    array[21] = tmp;
    tmp = vminf(array[22], array[70]);
    array[70] = vmaxf(array[22], array[70]);
    array[22] = tmp;
    tmp = vminf(array[23], array[71]);
    array[71] = vmaxf(array[23], array[71]);
    array[23] = tmp;
    tmp = vminf(array[24], array[72]);
    array[72] = vmaxf(array[24], array[72]);
    array[24] = tmp;
    tmp = vminf(array[25], array[73]);
    array[73] = vmaxf(array[25], array[73]);
    array[25] = tmp;
    tmp = vminf(array[26], array[74]);
    array[74] = vmaxf(array[26], array[74]);
    array[26] = tmp;
    tmp = vminf(array[27], array[75]);
    array[75] = vmaxf(array[27], array[75]);
    array[27] = tmp;
    tmp = vminf(array[28], array[76]);
    array[76] = vmaxf(array[28], array[76]);
    array[28] = tmp;
    tmp = vminf(array[29], array[77]);
    array[77] = vmaxf(array[29], array[77]);
    array[29] = tmp;
    tmp = vminf(array[30], array[78]);
    array[78] = vmaxf(array[30], array[78]);
    array[30] = tmp;
    tmp = vminf(array[31], array[79]);
    array[79] = vmaxf(array[31], array[79]);
    array[31] = tmp;
    tmp = vminf(array[16], array[32]);
    array[32] = vmaxf(array[16], array[32]);
    array[16] = tmp;
    tmp = vminf(array[17], array[33]);
    array[33] = vmaxf(array[17], array[33]);
    array[17] = tmp;
    tmp = vminf(array[18], array[34]);
    array[34] = vmaxf(array[18], array[34]);
    array[18] = tmp;
    tmp = vminf(array[19], array[35]);
    array[35] = vmaxf(array[19], array[35]);
    array[19] = tmp;
    tmp = vminf(array[20], array[36]);
    array[36] = vmaxf(array[20], array[36]);
    array[20] = tmp;
    tmp = vminf(array[21], array[37]);
    array[37] = vmaxf(array[21], array[37]);
    array[21] = tmp;
    tmp = vminf(array[22], array[38]);
    array[38] = vmaxf(array[22], array[38]);
    array[22] = tmp;
    tmp = vminf(array[23], array[39]);
    array[39] = vmaxf(array[23], array[39]);
    array[23] = tmp;
    tmp = vminf(array[24], array[40]);
    array[40] = vmaxf(array[24], array[40]);
    array[24] = tmp;
    tmp = vminf(array[25], array[41]);
    array[41] = vmaxf(array[25], array[41]);
    array[25] = tmp;
    tmp = vminf(array[26], array[42]);
    array[42] = vmaxf(array[26], array[42]);
    array[26] = tmp;
    tmp = vminf(array[27], array[43]);
    array[43] = vmaxf(array[27], array[43]);
    array[27] = tmp;
    tmp = vminf(array[28], array[44]);
    array[44] = vmaxf(array[28], array[44]);
    array[28] = tmp;
    tmp = vminf(array[29], array[45]);
    array[45] = vmaxf(array[29], array[45]);
    array[29] = tmp;
    tmp = vminf(array[30], array[46]);
    array[46] = vmaxf(array[30], array[46]);
    array[30] = tmp;
    tmp = vminf(array[31], array[47]);
    array[47] = vmaxf(array[31], array[47]);
    array[31] = tmp;
    tmp = vminf(array[48], array[64]);
    array[64] = vmaxf(array[48], array[64]);
    array[48] = tmp;
    tmp = vminf(array[49], array[65]);
    array[65] = vmaxf(array[49], array[65]);
    array[49] = tmp;
    tmp = vminf(array[50], array[66]);
    array[66] = vmaxf(array[50], array[66]);
    array[50] = tmp;
    tmp = vminf(array[51], array[67]);
    array[67] = vmaxf(array[51], array[67]);
    array[51] = tmp;
    tmp = vminf(array[52], array[68]);
    array[68] = vmaxf(array[52], array[68]);
    array[52] = tmp;
    tmp = vminf(array[53], array[69]);
    array[69] = vmaxf(array[53], array[69]);
    array[53] = tmp;
    tmp = vminf(array[54], array[70]);
    array[70] = vmaxf(array[54], array[70]);
    array[54] = tmp;
    tmp = vminf(array[55], array[71]);
    array[71] = vmaxf(array[55], array[71]);
    array[55] = tmp;
    tmp = vminf(array[56], array[72]);
    array[72] = vmaxf(array[56], array[72]);
    array[56] = tmp;
    tmp = vminf(array[57], array[73]);
    array[73] = vmaxf(array[57], array[73]);
    array[57] = tmp;
    tmp = vminf(array[58], array[74]);
    array[74] = vmaxf(array[58], array[74]);
    array[58] = tmp;
    tmp = vminf(array[59], array[75]);
    array[75] = vmaxf(array[59], array[75]);
    array[59] = tmp;
    tmp = vminf(array[60], array[76]);
    array[76] = vmaxf(array[60], array[76]);
    array[60] = tmp;
    tmp = vminf(array[61], array[77]);
    array[77] = vmaxf(array[61], array[77]);
    array[61] = tmp;
    tmp = vminf(array[62], array[78]);
    array[78] = vmaxf(array[62], array[78]);
    array[62] = tmp;
    tmp = vminf(array[63], array[79]);
    array[79] = vmaxf(array[63], array[79]);
    array[63] = tmp;
    tmp = vminf(array[0], array[8]);
    array[8] = vmaxf(array[0], array[8]);
    array[0] = tmp;
    tmp = vminf(array[1], array[9]);
    array[9] = vmaxf(array[1], array[9]);
    array[1] = tmp;
    tmp = vminf(array[2], array[10]);
    array[10] = vmaxf(array[2], array[10]);
    array[2] = tmp;
    tmp = vminf(array[3], array[11]);
    array[11] = vmaxf(array[3], array[11]);
    array[3] = tmp;
    tmp = vminf(array[4], array[12]);
    array[12] = vmaxf(array[4], array[12]);
    array[4] = tmp;
    tmp = vminf(array[5], array[13]);
    array[13] = vmaxf(array[5], array[13]);
    array[5] = tmp;
    tmp = vminf(array[6], array[14]);
    array[14] = vmaxf(array[6], array[14]);
    array[6] = tmp;
    tmp = vminf(array[7], array[15]);
    array[15] = vmaxf(array[7], array[15]);
    array[7] = tmp;
    tmp = vminf(array[16], array[24]);
    array[24] = vmaxf(array[16], array[24]);
    array[16] = tmp;
    tmp = vminf(array[17], array[25]);
    array[25] = vmaxf(array[17], array[25]);
    array[17] = tmp;
    tmp = vminf(array[18], array[26]);
    array[26] = vmaxf(array[18], array[26]);
    array[18] = tmp;
    tmp = vminf(array[19], array[27]);
    array[27] = vmaxf(array[19], array[27]);
    array[19] = tmp;
    tmp = vminf(array[20], array[28]);
    array[28] = vmaxf(array[20], array[28]);
    array[20] = tmp;
    tmp = vminf(array[21], array[29]);
    array[29] = vmaxf(array[21], array[29]);
    array[21] = tmp;
    tmp = vminf(array[22], array[30]);
    array[30] = vmaxf(array[22], array[30]);
    array[22] = tmp;
    tmp = vminf(array[23], array[31]);
    array[31] = vmaxf(array[23], array[31]);
    array[23] = tmp;
    tmp = vminf(array[32], array[40]);
    array[40] = vmaxf(array[32], array[40]);
    array[32] = tmp;
    tmp = vminf(array[33], array[41]);
    array[41] = vmaxf(array[33], array[41]);
    array[33] = tmp;
    tmp = vminf(array[34], array[42]);
    array[42] = vmaxf(array[34], array[42]);
    array[34] = tmp;
    tmp = vminf(array[35], array[43]);
    array[43] = vmaxf(array[35], array[43]);
    array[35] = tmp;
    tmp = vminf(array[36], array[44]);
    array[44] = vmaxf(array[36], array[44]);
    array[36] = tmp;
    tmp = vminf(array[37], array[45]);
    array[45] = vmaxf(array[37], array[45]);
    array[37] = tmp;
    tmp = vminf(array[38], array[46]);
    array[46] = vmaxf(array[38], array[46]);
    array[38] = tmp;
    tmp = vminf(array[39], array[47]);
    array[47] = vmaxf(array[39], array[47]);
    array[39] = tmp;
    tmp = vminf(array[48], array[56]);
    array[56] = vmaxf(array[48], array[56]);
    array[48] = tmp;
    tmp = vminf(array[49], array[57]);
    array[57] = vmaxf(array[49], array[57]);
    array[49] = tmp;
    tmp = vminf(array[50], array[58]);
    array[58] = vmaxf(array[50], array[58]);
    array[50] = tmp;
    tmp = vminf(array[51], array[59]);
    array[59] = vmaxf(array[51], array[59]);
    array[51] = tmp;
    tmp = vminf(array[52], array[60]);
    array[60] = vmaxf(array[52], array[60]);
    array[52] = tmp;
    tmp = vminf(array[53], array[61]);
    array[61] = vmaxf(array[53], array[61]);
    array[53] = tmp;
    tmp = vminf(array[54], array[62]);
    array[62] = vmaxf(array[54], array[62]);
    array[54] = tmp;
    tmp = vminf(array[55], array[63]);
    array[63] = vmaxf(array[55], array[63]);
    array[55] = tmp;
    tmp = vminf(array[64], array[72]);
    array[72] = vmaxf(array[64], array[72]);
    array[64] = tmp;
    tmp = vminf(array[65], array[73]);
    array[73] = vmaxf(array[65], array[73]);
    array[65] = tmp;
    tmp = vminf(array[66], array[74]);
    array[74] = vmaxf(array[66], array[74]);
    array[66] = tmp;
    tmp = vminf(array[67], array[75]);
    array[75] = vmaxf(array[67], array[75]);
    array[67] = tmp;
    tmp = vminf(array[68], array[76]);
    array[76] = vmaxf(array[68], array[76]);
    array[68] = tmp;
    tmp = vminf(array[69], array[77]);
    array[77] = vmaxf(array[69], array[77]);
    array[69] = tmp;
    tmp = vminf(array[70], array[78]);
    array[78] = vmaxf(array[70], array[78]);
    array[70] = tmp;
    tmp = vminf(array[71], array[79]);
    array[79] = vmaxf(array[71], array[79]);
    array[71] = tmp;
    tmp = vminf(array[8], array[64]);
    array[64] = vmaxf(array[8], array[64]);
    array[8] = tmp;
    tmp = vminf(array[9], array[65]);
    array[65] = vmaxf(array[9], array[65]);
    array[9] = tmp;
    tmp = vminf(array[10], array[66]);
    array[66] = vmaxf(array[10], array[66]);
    array[10] = tmp;
    tmp = vminf(array[11], array[67]);
    array[67] = vmaxf(array[11], array[67]);
    array[11] = tmp;
    tmp = vminf(array[12], array[68]);
    array[68] = vmaxf(array[12], array[68]);
    array[12] = tmp;
    tmp = vminf(array[13], array[69]);
    array[69] = vmaxf(array[13], array[69]);
    array[13] = tmp;
    tmp = vminf(array[14], array[70]);
    array[70] = vmaxf(array[14], array[70]);
    array[14] = tmp;
    tmp = vminf(array[15], array[71]);
    array[71] = vmaxf(array[15], array[71]);
    array[15] = tmp;
    tmp = vminf(array[24], array[80]);
    array[80] = vmaxf(array[24], array[80]);
    array[24] = tmp;
    tmp = vminf(array[8], array[32]);
    array[32] = vmaxf(array[8], array[32]);
    array[8] = tmp;
    tmp = vminf(array[9], array[33]);
    array[33] = vmaxf(array[9], array[33]);
    array[9] = tmp;
    tmp = vminf(array[10], array[34]);
    array[34] = vmaxf(array[10], array[34]);
    array[10] = tmp;
    tmp = vminf(array[11], array[35]);
    array[35] = vmaxf(array[11], array[35]);
    array[11] = tmp;
    tmp = vminf(array[12], array[36]);
    array[36] = vmaxf(array[12], array[36]);
    array[12] = tmp;
    tmp = vminf(array[13], array[37]);
    array[37] = vmaxf(array[13], array[37]);
    array[13] = tmp;
    tmp = vminf(array[14], array[38]);
    array[38] = vmaxf(array[14], array[38]);
    array[14] = tmp;
    tmp = vminf(array[15], array[39]);
    array[39] = vmaxf(array[15], array[39]);
    array[15] = tmp;
    tmp = vminf(array[24], array[48]);
    array[48] = vmaxf(array[24], array[48]);
    array[24] = tmp;
    tmp = vminf(array[25], array[49]);
    array[49] = vmaxf(array[25], array[49]);
    array[25] = tmp;
    tmp = vminf(array[26], array[50]);
    array[50] = vmaxf(array[26], array[50]);
    array[26] = tmp;
    tmp = vminf(array[27], array[51]);
    array[51] = vmaxf(array[27], array[51]);
    array[27] = tmp;
    tmp = vminf(array[28], array[52]);
    array[52] = vmaxf(array[28], array[52]);
    array[28] = tmp;
    tmp = vminf(array[29], array[53]);
    array[53] = vmaxf(array[29], array[53]);
    array[29] = tmp;
    tmp = vminf(array[30], array[54]);
    array[54] = vmaxf(array[30], array[54]);
    array[30] = tmp;
    tmp = vminf(array[31], array[55]);
    array[55] = vmaxf(array[31], array[55]);
    array[31] = tmp;
    tmp = vminf(array[40], array[64]);
    array[64] = vmaxf(array[40], array[64]);
    array[40] = tmp;
    tmp = vminf(array[41], array[65]);
    array[65] = vmaxf(array[41], array[65]);
    array[41] = tmp;
    tmp = vminf(array[42], array[66]);
    array[66] = vmaxf(array[42], array[66]);
    array[42] = tmp;
    tmp = vminf(array[43], array[67]);
    array[67] = vmaxf(array[43], array[67]);
    array[43] = tmp;
    tmp = vminf(array[44], array[68]);
    array[68] = vmaxf(array[44], array[68]);
    array[44] = tmp;
    tmp = vminf(array[45], array[69]);
    array[69] = vmaxf(array[45], array[69]);
    array[45] = tmp;
    tmp = vminf(array[46], array[70]);
    array[70] = vmaxf(array[46], array[70]);
    array[46] = tmp;
    tmp = vminf(array[47], array[71]);
    array[71] = vmaxf(array[47], array[71]);
    array[47] = tmp;
    tmp = vminf(array[56], array[80]);
    array[80] = vmaxf(array[56], array[80]);
    array[56] = tmp;
    tmp = vminf(array[8], array[16]);
    array[16] = vmaxf(array[8], array[16]);
    array[8] = tmp;
    tmp = vminf(array[9], array[17]);
    array[17] = vmaxf(array[9], array[17]);
    array[9] = tmp;
    tmp = vminf(array[10], array[18]);
    array[18] = vmaxf(array[10], array[18]);
    array[10] = tmp;
    tmp = vminf(array[11], array[19]);
    array[19] = vmaxf(array[11], array[19]);
    array[11] = tmp;
    tmp = vminf(array[12], array[20]);
    array[20] = vmaxf(array[12], array[20]);
    array[12] = tmp;
    tmp = vminf(array[13], array[21]);
    array[21] = vmaxf(array[13], array[21]);
    array[13] = tmp;
    tmp = vminf(array[14], array[22]);
    array[22] = vmaxf(array[14], array[22]);
    array[14] = tmp;
    tmp = vminf(array[15], array[23]);
    array[23] = vmaxf(array[15], array[23]);
    array[15] = tmp;
    tmp = vminf(array[24], array[32]);
    array[32] = vmaxf(array[24], array[32]);
    array[24] = tmp;
    tmp = vminf(array[25], array[33]);
    array[33] = vmaxf(array[25], array[33]);
    array[25] = tmp;
    tmp = vminf(array[26], array[34]);
    array[34] = vmaxf(array[26], array[34]);
    array[26] = tmp;
    tmp = vminf(array[27], array[35]);
    array[35] = vmaxf(array[27], array[35]);
    array[27] = tmp;
    tmp = vminf(array[28], array[36]);
    array[36] = vmaxf(array[28], array[36]);
    array[28] = tmp;
    tmp = vminf(array[29], array[37]);
    array[37] = vmaxf(array[29], array[37]);
    array[29] = tmp;
    tmp = vminf(array[30], array[38]);
    array[38] = vmaxf(array[30], array[38]);
    array[30] = tmp;
    tmp = vminf(array[31], array[39]);
    array[39] = vmaxf(array[31], array[39]);
    array[31] = tmp;
    tmp = vminf(array[40], array[48]);
    array[48] = vmaxf(array[40], array[48]);
    array[40] = tmp;
    tmp = vminf(array[41], array[49]);
    array[49] = vmaxf(array[41], array[49]);
    array[41] = tmp;
    tmp = vminf(array[42], array[50]);
    array[50] = vmaxf(array[42], array[50]);
    array[42] = tmp;
    tmp = vminf(array[43], array[51]);
    array[51] = vmaxf(array[43], array[51]);
    array[43] = tmp;
    tmp = vminf(array[44], array[52]);
    array[52] = vmaxf(array[44], array[52]);
    array[44] = tmp;
    tmp = vminf(array[45], array[53]);
    array[53] = vmaxf(array[45], array[53]);
    array[45] = tmp;
    tmp = vminf(array[46], array[54]);
    array[54] = vmaxf(array[46], array[54]);
    array[46] = tmp;
    tmp = vminf(array[47], array[55]);
    array[55] = vmaxf(array[47], array[55]);
    array[47] = tmp;
    tmp = vminf(array[56], array[64]);
    array[64] = vmaxf(array[56], array[64]);
    array[56] = tmp;
    tmp = vminf(array[57], array[65]);
    array[65] = vmaxf(array[57], array[65]);
    array[57] = tmp;
    tmp = vminf(array[58], array[66]);
    array[66] = vmaxf(array[58], array[66]);
    array[58] = tmp;
    tmp = vminf(array[59], array[67]);
    array[67] = vmaxf(array[59], array[67]);
    array[59] = tmp;
    tmp = vminf(array[60], array[68]);
    array[68] = vmaxf(array[60], array[68]);
    array[60] = tmp;
    tmp = vminf(array[61], array[69]);
    array[69] = vmaxf(array[61], array[69]);
    array[61] = tmp;
    tmp = vminf(array[62], array[70]);
    array[70] = vmaxf(array[62], array[70]);
    array[62] = tmp;
    tmp = vminf(array[63], array[71]);
    array[71] = vmaxf(array[63], array[71]);
    array[63] = tmp;
    tmp = vminf(array[72], array[80]);
    array[80] = vmaxf(array[72], array[80]);
    array[72] = tmp;
    tmp = vminf(array[0], array[4]);
    array[4] = vmaxf(array[0], array[4]);
    array[0] = tmp;
    tmp = vminf(array[1], array[5]);
    array[5] = vmaxf(array[1], array[5]);
    array[1] = tmp;
    tmp = vminf(array[2], array[6]);
    array[6] = vmaxf(array[2], array[6]);
    array[2] = tmp;
    tmp = vminf(array[3], array[7]);
    array[7] = vmaxf(array[3], array[7]);
    array[3] = tmp;
    tmp = vminf(array[8], array[12]);
    array[12] = vmaxf(array[8], array[12]);
    array[8] = tmp;
    tmp = vminf(array[9], array[13]);
    array[13] = vmaxf(array[9], array[13]);
    array[9] = tmp;
    tmp = vminf(array[10], array[14]);
    array[14] = vmaxf(array[10], array[14]);
    array[10] = tmp;
    tmp = vminf(array[11], array[15]);
    array[15] = vmaxf(array[11], array[15]);
    array[11] = tmp;
    tmp = vminf(array[16], array[20]);
    array[20] = vmaxf(array[16], array[20]);
    array[16] = tmp;
    tmp = vminf(array[17], array[21]);
    array[21] = vmaxf(array[17], array[21]);
    array[17] = tmp;
    tmp = vminf(array[18], array[22]);
    array[22] = vmaxf(array[18], array[22]);
    array[18] = tmp;
    tmp = vminf(array[19], array[23]);
    array[23] = vmaxf(array[19], array[23]);
    array[19] = tmp;
    tmp = vminf(array[24], array[28]);
    array[28] = vmaxf(array[24], array[28]);
    array[24] = tmp;
    tmp = vminf(array[25], array[29]);
    array[29] = vmaxf(array[25], array[29]);
    array[25] = tmp;
    tmp = vminf(array[26], array[30]);
    array[30] = vmaxf(array[26], array[30]);
    array[26] = tmp;
    tmp = vminf(array[27], array[31]);
    array[31] = vmaxf(array[27], array[31]);
    array[27] = tmp;
    tmp = vminf(array[32], array[36]);
    array[36] = vmaxf(array[32], array[36]);
    array[32] = tmp;
    tmp = vminf(array[33], array[37]);
    array[37] = vmaxf(array[33], array[37]);
    array[33] = tmp;
    tmp = vminf(array[34], array[38]);
    array[38] = vmaxf(array[34], array[38]);
    array[34] = tmp;
    tmp = vminf(array[35], array[39]);
    array[39] = vmaxf(array[35], array[39]);
    array[35] = tmp;
    tmp = vminf(array[40], array[44]);
    array[44] = vmaxf(array[40], array[44]);
    array[40] = tmp;
    tmp = vminf(array[41], array[45]);
    array[45] = vmaxf(array[41], array[45]);
    array[41] = tmp;
    tmp = vminf(array[42], array[46]);
    array[46] = vmaxf(array[42], array[46]);
    array[42] = tmp;
    tmp = vminf(array[43], array[47]);
    array[47] = vmaxf(array[43], array[47]);
    array[43] = tmp;
    tmp = vminf(array[48], array[52]);
    array[52] = vmaxf(array[48], array[52]);
    array[48] = tmp;
    tmp = vminf(array[49], array[53]);
    array[53] = vmaxf(array[49], array[53]);
    array[49] = tmp;
    tmp = vminf(array[50], array[54]);
    array[54] = vmaxf(array[50], array[54]);
    array[50] = tmp;
    tmp = vminf(array[51], array[55]);
    array[55] = vmaxf(array[51], array[55]);
    array[51] = tmp;
    tmp = vminf(array[56], array[60]);
    array[60] = vmaxf(array[56], array[60]);
    array[56] = tmp;
    tmp = vminf(array[57], array[61]);
    array[61] = vmaxf(array[57], array[61]);
    array[57] = tmp;
    tmp = vminf(array[58], array[62]);
    array[62] = vmaxf(array[58], array[62]);
    array[58] = tmp;
    tmp = vminf(array[59], array[63]);
    array[63] = vmaxf(array[59], array[63]);
    array[59] = tmp;
    tmp = vminf(array[64], array[68]);
    array[68] = vmaxf(array[64], array[68]);
    array[64] = tmp;
    tmp = vminf(array[65], array[69]);
    array[69] = vmaxf(array[65], array[69]);
    array[65] = tmp;
    tmp = vminf(array[66], array[70]);
    array[70] = vmaxf(array[66], array[70]);
    array[66] = tmp;
    tmp = vminf(array[67], array[71]);
    array[71] = vmaxf(array[67], array[71]);
    array[67] = tmp;
    tmp = vminf(array[72], array[76]);
    array[76] = vmaxf(array[72], array[76]);
    array[72] = tmp;
    tmp = vminf(array[73], array[77]);
    array[77] = vmaxf(array[73], array[77]);
    array[73] = tmp;
    tmp = vminf(array[74], array[78]);
    array[78] = vmaxf(array[74], array[78]);
    array[74] = tmp;
    tmp = vminf(array[75], array[79]);
    array[79] = vmaxf(array[75], array[79]);
    array[75] = tmp;
    tmp = vminf(array[4], array[64]);
    array[64] = vmaxf(array[4], array[64]);
    array[4] = tmp;
    tmp = vminf(array[5], array[65]);
    array[65] = vmaxf(array[5], array[65]);
    array[5] = tmp;
    tmp = vminf(array[6], array[66]);
    array[66] = vmaxf(array[6], array[66]);
    array[6] = tmp;
    tmp = vminf(array[7], array[67]);
    array[67] = vmaxf(array[7], array[67]);
    array[7] = tmp;
    tmp = vminf(array[12], array[72]);
    array[72] = vmaxf(array[12], array[72]);
    array[12] = tmp;
    tmp = vminf(array[13], array[73]);
    array[73] = vmaxf(array[13], array[73]);
    array[13] = tmp;
    tmp = vminf(array[14], array[74]);
    array[74] = vmaxf(array[14], array[74]);
    array[14] = tmp;
    tmp = vminf(array[15], array[75]);
    array[75] = vmaxf(array[15], array[75]);
    array[15] = tmp;
    tmp = vminf(array[20], array[80]);
    array[80] = vmaxf(array[20], array[80]);
    array[20] = tmp;
    tmp = vminf(array[4], array[32]);
    array[32] = vmaxf(array[4], array[32]);
    array[4] = tmp;
    tmp = vminf(array[5], array[33]);
    array[33] = vmaxf(array[5], array[33]);
    array[5] = tmp;
    tmp = vminf(array[6], array[34]);
    array[34] = vmaxf(array[6], array[34]);
    array[6] = tmp;
    tmp = vminf(array[7], array[35]);
    array[35] = vmaxf(array[7], array[35]);
    array[7] = tmp;
    tmp = vminf(array[12], array[40]);
    array[40] = vmaxf(array[12], array[40]);
    array[12] = tmp;
    tmp = vminf(array[13], array[41]);
    array[41] = vmaxf(array[13], array[41]);
    array[13] = tmp;
    tmp = vminf(array[14], array[42]);
    array[42] = vmaxf(array[14], array[42]);
    array[14] = tmp;
    tmp = vminf(array[15], array[43]);
    array[43] = vmaxf(array[15], array[43]);
    array[15] = tmp;
    tmp = vminf(array[20], array[48]);
    array[48] = vmaxf(array[20], array[48]);
    array[20] = tmp;
    tmp = vminf(array[21], array[49]);
    array[49] = vmaxf(array[21], array[49]);
    array[21] = tmp;
    tmp = vminf(array[22], array[50]);
    array[50] = vmaxf(array[22], array[50]);
    array[22] = tmp;
    tmp = vminf(array[23], array[51]);
    array[51] = vmaxf(array[23], array[51]);
    array[23] = tmp;
    tmp = vminf(array[28], array[56]);
    array[56] = vmaxf(array[28], array[56]);
    array[28] = tmp;
    tmp = vminf(array[29], array[57]);
    array[57] = vmaxf(array[29], array[57]);
    array[29] = tmp;
    tmp = vminf(array[30], array[58]);
    array[58] = vmaxf(array[30], array[58]);
    array[30] = tmp;
    tmp = vminf(array[31], array[59]);
    array[59] = vmaxf(array[31], array[59]);
    array[31] = tmp;
    tmp = vminf(array[36], array[64]);
    array[64] = vmaxf(array[36], array[64]);
    array[36] = tmp;
    tmp = vminf(array[37], array[65]);
    array[65] = vmaxf(array[37], array[65]);
    array[37] = tmp;
    tmp = vminf(array[38], array[66]);
    array[66] = vmaxf(array[38], array[66]);
    array[38] = tmp;
    tmp = vminf(array[39], array[67]);
    array[67] = vmaxf(array[39], array[67]);
    array[39] = tmp;
    tmp = vminf(array[44], array[72]);
    array[72] = vmaxf(array[44], array[72]);
    array[44] = tmp;
    tmp = vminf(array[45], array[73]);
    array[73] = vmaxf(array[45], array[73]);
    array[45] = tmp;
    tmp = vminf(array[46], array[74]);
    array[74] = vmaxf(array[46], array[74]);
    array[46] = tmp;
    tmp = vminf(array[47], array[75]);
    array[75] = vmaxf(array[47], array[75]);
    array[47] = tmp;
    tmp = vminf(array[52], array[80]);
    array[80] = vmaxf(array[52], array[80]);
    array[52] = tmp;
    tmp = vminf(array[4], array[16]);
    array[16] = vmaxf(array[4], array[16]);
    array[4] = tmp;
    tmp = vminf(array[5], array[17]);
    array[17] = vmaxf(array[5], array[17]);
    array[5] = tmp;
    tmp = vminf(array[6], array[18]);
    array[18] = vmaxf(array[6], array[18]);
    array[6] = tmp;
    tmp = vminf(array[7], array[19]);
    array[19] = vmaxf(array[7], array[19]);
    array[7] = tmp;
    tmp = vminf(array[12], array[24]);
    array[24] = vmaxf(array[12], array[24]);
    array[12] = tmp;
    tmp = vminf(array[13], array[25]);
    array[25] = vmaxf(array[13], array[25]);
    array[13] = tmp;
    tmp = vminf(array[14], array[26]);
    array[26] = vmaxf(array[14], array[26]);
    array[14] = tmp;
    tmp = vminf(array[15], array[27]);
    array[27] = vmaxf(array[15], array[27]);
    array[15] = tmp;
    tmp = vminf(array[20], array[32]);
    array[32] = vmaxf(array[20], array[32]);
    array[20] = tmp;
    tmp = vminf(array[21], array[33]);
    array[33] = vmaxf(array[21], array[33]);
    array[21] = tmp;
    tmp = vminf(array[22], array[34]);
    array[34] = vmaxf(array[22], array[34]);
    array[22] = tmp;
    tmp = vminf(array[23], array[35]);
    array[35] = vmaxf(array[23], array[35]);
    array[23] = tmp;
    tmp = vminf(array[28], array[40]);
    array[40] = vmaxf(array[28], array[40]);
    array[28] = tmp;
    tmp = vminf(array[29], array[41]);
    array[41] = vmaxf(array[29], array[41]);
    array[29] = tmp;
    tmp = vminf(array[30], array[42]);
    array[42] = vmaxf(array[30], array[42]);
    array[30] = tmp;
    tmp = vminf(array[31], array[43]);
    array[43] = vmaxf(array[31], array[43]);
    array[31] = tmp;
    tmp = vminf(array[36], array[48]);
    array[48] = vmaxf(array[36], array[48]);
    array[36] = tmp;
    tmp = vminf(array[37], array[49]);
    array[49] = vmaxf(array[37], array[49]);
    array[37] = tmp;
    tmp = vminf(array[38], array[50]);
    array[50] = vmaxf(array[38], array[50]);
    array[38] = tmp;
    tmp = vminf(array[39], array[51]);
    array[51] = vmaxf(array[39], array[51]);
    array[39] = tmp;
    tmp = vminf(array[44], array[56]);
    array[56] = vmaxf(array[44], array[56]);
    array[44] = tmp;
    tmp = vminf(array[45], array[57]);
    array[57] = vmaxf(array[45], array[57]);
    array[45] = tmp;
    tmp = vminf(array[46], array[58]);
    array[58] = vmaxf(array[46], array[58]);
    array[46] = tmp;
    tmp = vminf(array[47], array[59]);
    array[59] = vmaxf(array[47], array[59]);
    array[47] = tmp;
    tmp = vminf(array[52], array[64]);
    array[64] = vmaxf(array[52], array[64]);
    array[52] = tmp;
    tmp = vminf(array[53], array[65]);
    array[65] = vmaxf(array[53], array[65]);
    array[53] = tmp;
    tmp = vminf(array[54], array[66]);
    array[66] = vmaxf(array[54], array[66]);
    array[54] = tmp;
    tmp = vminf(array[55], array[67]);
    array[67] = vmaxf(array[55], array[67]);
    array[55] = tmp;
    tmp = vminf(array[60], array[72]);
    array[72] = vmaxf(array[60], array[72]);
    array[60] = tmp;
    tmp = vminf(array[61], array[73]);
    array[73] = vmaxf(array[61], array[73]);
    array[61] = tmp;
    tmp = vminf(array[62], array[74]);
    array[74] = vmaxf(array[62], array[74]);
    array[62] = tmp;
    tmp = vminf(array[63], array[75]);
    array[75] = vmaxf(array[63], array[75]);
    array[63] = tmp;
    tmp = vminf(array[68], array[80]);
    array[80] = vmaxf(array[68], array[80]);
    array[68] = tmp;
    tmp = vminf(array[4], array[8]);
    array[8] = vmaxf(array[4], array[8]);
    array[4] = tmp;
    tmp = vminf(array[5], array[9]);
    array[9] = vmaxf(array[5], array[9]);
    array[5] = tmp;
    tmp = vminf(array[6], array[10]);
    array[10] = vmaxf(array[6], array[10]);
    array[6] = tmp;
    tmp = vminf(array[7], array[11]);
    array[11] = vmaxf(array[7], array[11]);
    array[7] = tmp;
    tmp = vminf(array[12], array[16]);
    array[16] = vmaxf(array[12], array[16]);
    array[12] = tmp;
    tmp = vminf(array[13], array[17]);
    array[17] = vmaxf(array[13], array[17]);
    array[13] = tmp;
    tmp = vminf(array[14], array[18]);
    array[18] = vmaxf(array[14], array[18]);
    array[14] = tmp;
    tmp = vminf(array[15], array[19]);
    array[19] = vmaxf(array[15], array[19]);
    array[15] = tmp;
    tmp = vminf(array[20], array[24]);
    array[24] = vmaxf(array[20], array[24]);
    array[20] = tmp;
    tmp = vminf(array[21], array[25]);
    array[25] = vmaxf(array[21], array[25]);
    array[21] = tmp;
    tmp = vminf(array[22], array[26]);
    array[26] = vmaxf(array[22], array[26]);
    array[22] = tmp;
    tmp = vminf(array[23], array[27]);
    array[27] = vmaxf(array[23], array[27]);
    array[23] = tmp;
    tmp = vminf(array[28], array[32]);
    array[32] = vmaxf(array[28], array[32]);
    array[28] = tmp;
    tmp = vminf(array[29], array[33]);
    array[33] = vmaxf(array[29], array[33]);
    array[29] = tmp;
    tmp = vminf(array[30], array[34]);
    array[34] = vmaxf(array[30], array[34]);
    array[30] = tmp;
    tmp = vminf(array[31], array[35]);
    array[35] = vmaxf(array[31], array[35]);
    array[31] = tmp;
    tmp = vminf(array[36], array[40]);
    array[40] = vmaxf(array[36], array[40]);
    array[36] = tmp;
    tmp = vminf(array[37], array[41]);
    array[41] = vmaxf(array[37], array[41]);
    array[37] = tmp;
    tmp = vminf(array[38], array[42]);
    array[42] = vmaxf(array[38], array[42]);
    array[38] = tmp;
    tmp = vminf(array[39], array[43]);
    array[43] = vmaxf(array[39], array[43]);
    array[39] = tmp;
    tmp = vminf(array[44], array[48]);
    array[48] = vmaxf(array[44], array[48]);
    array[44] = tmp;
    tmp = vminf(array[45], array[49]);
    array[49] = vmaxf(array[45], array[49]);
    array[45] = tmp;
    tmp = vminf(array[46], array[50]);
    array[50] = vmaxf(array[46], array[50]);
    array[46] = tmp;
    tmp = vminf(array[47], array[51]);
    array[51] = vmaxf(array[47], array[51]);
    array[47] = tmp;
    tmp = vminf(array[52], array[56]);
    array[56] = vmaxf(array[52], array[56]);
    array[52] = tmp;
    tmp = vminf(array[53], array[57]);
    array[57] = vmaxf(array[53], array[57]);
    array[53] = tmp;
    tmp = vminf(array[54], array[58]);
    array[58] = vmaxf(array[54], array[58]);
    array[54] = tmp;
    tmp = vminf(array[55], array[59]);
    array[59] = vmaxf(array[55], array[59]);
    array[55] = tmp;
    tmp = vminf(array[60], array[64]);
    array[64] = vmaxf(array[60], array[64]);
    array[60] = tmp;
    tmp = vminf(array[61], array[65]);
    array[65] = vmaxf(array[61], array[65]);
    array[61] = tmp;
    tmp = vminf(array[62], array[66]);
    array[66] = vmaxf(array[62], array[66]);
    array[62] = tmp;
    tmp = vminf(array[63], array[67]);
    array[67] = vmaxf(array[63], array[67]);
    array[63] = tmp;
    tmp = vminf(array[68], array[72]);
    array[72] = vmaxf(array[68], array[72]);
    array[68] = tmp;
    tmp = vminf(array[69], array[73]);
    array[73] = vmaxf(array[69], array[73]);
    array[69] = tmp;
    tmp = vminf(array[70], array[74]);
    array[74] = vmaxf(array[70], array[74]);
    array[70] = tmp;
    tmp = vminf(array[71], array[75]);
    array[75] = vmaxf(array[71], array[75]);
    array[71] = tmp;
    tmp = vminf(array[76], array[80]);
    array[80] = vmaxf(array[76], array[80]);
    array[76] = tmp;
    tmp = vminf(array[0], array[2]);
    array[2] = vmaxf(array[0], array[2]);
    array[0] = tmp;
    tmp = vminf(array[1], array[3]);
    array[3] = vmaxf(array[1], array[3]);
    array[1] = tmp;
    tmp = vminf(array[4], array[6]);
    array[6] = vmaxf(array[4], array[6]);
    array[4] = tmp;
    tmp = vminf(array[5], array[7]);
    array[7] = vmaxf(array[5], array[7]);
    array[5] = tmp;
    tmp = vminf(array[8], array[10]);
    array[10] = vmaxf(array[8], array[10]);
    array[8] = tmp;
    tmp = vminf(array[9], array[11]);
    array[11] = vmaxf(array[9], array[11]);
    array[9] = tmp;
    tmp = vminf(array[12], array[14]);
    array[14] = vmaxf(array[12], array[14]);
    array[12] = tmp;
    tmp = vminf(array[13], array[15]);
    array[15] = vmaxf(array[13], array[15]);
    array[13] = tmp;
    tmp = vminf(array[16], array[18]);
    array[18] = vmaxf(array[16], array[18]);
    array[16] = tmp;
    tmp = vminf(array[17], array[19]);
    array[19] = vmaxf(array[17], array[19]);
    array[17] = tmp;
    tmp = vminf(array[20], array[22]);
    array[22] = vmaxf(array[20], array[22]);
    array[20] = tmp;
    tmp = vminf(array[21], array[23]);
    array[23] = vmaxf(array[21], array[23]);
    array[21] = tmp;
    tmp = vminf(array[24], array[26]);
    array[26] = vmaxf(array[24], array[26]);
    array[24] = tmp;
    tmp = vminf(array[25], array[27]);
    array[27] = vmaxf(array[25], array[27]);
    array[25] = tmp;
    tmp = vminf(array[28], array[30]);
    array[30] = vmaxf(array[28], array[30]);
    array[28] = tmp;
    tmp = vminf(array[29], array[31]);
    array[31] = vmaxf(array[29], array[31]);
    array[29] = tmp;
    tmp = vminf(array[32], array[34]);
    array[34] = vmaxf(array[32], array[34]);
    array[32] = tmp;
    tmp = vminf(array[33], array[35]);
    array[35] = vmaxf(array[33], array[35]);
    array[33] = tmp;
    tmp = vminf(array[36], array[38]);
    array[38] = vmaxf(array[36], array[38]);
    array[36] = tmp;
    tmp = vminf(array[37], array[39]);
    array[39] = vmaxf(array[37], array[39]);
    array[37] = tmp;
    tmp = vminf(array[40], array[42]);
    array[42] = vmaxf(array[40], array[42]);
    array[40] = tmp;
    tmp = vminf(array[41], array[43]);
    array[43] = vmaxf(array[41], array[43]);
    array[41] = tmp;
    tmp = vminf(array[44], array[46]);
    array[46] = vmaxf(array[44], array[46]);
    array[44] = tmp;
    tmp = vminf(array[45], array[47]);
    array[47] = vmaxf(array[45], array[47]);
    array[45] = tmp;
    tmp = vminf(array[48], array[50]);
    array[50] = vmaxf(array[48], array[50]);
    array[48] = tmp;
    tmp = vminf(array[49], array[51]);
    array[51] = vmaxf(array[49], array[51]);
    array[49] = tmp;
    tmp = vminf(array[52], array[54]);
    array[54] = vmaxf(array[52], array[54]);
    array[52] = tmp;
    tmp = vminf(array[53], array[55]);
    array[55] = vmaxf(array[53], array[55]);
    array[53] = tmp;
    tmp = vminf(array[56], array[58]);
    array[58] = vmaxf(array[56], array[58]);
    array[56] = tmp;
    tmp = vminf(array[57], array[59]);
    array[59] = vmaxf(array[57], array[59]);
    array[57] = tmp;
    tmp = vminf(array[60], array[62]);
    array[62] = vmaxf(array[60], array[62]);
    array[60] = tmp;
    tmp = vminf(array[61], array[63]);
    array[63] = vmaxf(array[61], array[63]);
    array[61] = tmp;
    tmp = vminf(array[64], array[66]);
    array[66] = vmaxf(array[64], array[66]);
    array[64] = tmp;
    tmp = vminf(array[65], array[67]);
    array[67] = vmaxf(array[65], array[67]);
    array[65] = tmp;
    tmp = vminf(array[68], array[70]);
    array[70] = vmaxf(array[68], array[70]);
    array[68] = tmp;
    tmp = vminf(array[69], array[71]);
    array[71] = vmaxf(array[69], array[71]);
    array[69] = tmp;
    tmp = vminf(array[72], array[74]);
    array[74] = vmaxf(array[72], array[74]);
    array[72] = tmp;
    tmp = vminf(array[73], array[75]);
    array[75] = vmaxf(array[73], array[75]);
    array[73] = tmp;
    tmp = vminf(array[76], array[78]);
    array[78] = vmaxf(array[76], array[78]);
    array[76] = tmp;
    tmp = vminf(array[77], array[79]);
    array[79] = vmaxf(array[77], array[79]);
    array[77] = tmp;
    tmp = vminf(array[2], array[64]);
    array[64] = vmaxf(array[2], array[64]);
    array[2] = tmp;
    tmp = vminf(array[3], array[65]);
    array[65] = vmaxf(array[3], array[65]);
    array[3] = tmp;
    tmp = vminf(array[6], array[68]);
    array[68] = vmaxf(array[6], array[68]);
    array[6] = tmp;
    tmp = vminf(array[7], array[69]);
    array[69] = vmaxf(array[7], array[69]);
    array[7] = tmp;
    tmp = vminf(array[10], array[72]);
    array[72] = vmaxf(array[10], array[72]);
    array[10] = tmp;
    tmp = vminf(array[11], array[73]);
    array[73] = vmaxf(array[11], array[73]);
    array[11] = tmp;
    tmp = vminf(array[14], array[76]);
    array[76] = vmaxf(array[14], array[76]);
    array[14] = tmp;
    tmp = vminf(array[15], array[77]);
    array[77] = vmaxf(array[15], array[77]);
    array[15] = tmp;
    tmp = vminf(array[18], array[80]);
    array[80] = vmaxf(array[18], array[80]);
    array[18] = tmp;
    tmp = vminf(array[2], array[32]);
    array[32] = vmaxf(array[2], array[32]);
    array[2] = tmp;
    tmp = vminf(array[3], array[33]);
    array[33] = vmaxf(array[3], array[33]);
    array[3] = tmp;
    tmp = vminf(array[6], array[36]);
    array[36] = vmaxf(array[6], array[36]);
    array[6] = tmp;
    tmp = vminf(array[7], array[37]);
    array[37] = vmaxf(array[7], array[37]);
    array[7] = tmp;
    tmp = vminf(array[10], array[40]);
    array[40] = vmaxf(array[10], array[40]);
    array[10] = tmp;
    tmp = vminf(array[11], array[41]);
    array[41] = vmaxf(array[11], array[41]);
    array[11] = tmp;
    tmp = vminf(array[14], array[44]);
    array[44] = vmaxf(array[14], array[44]);
    array[14] = tmp;
    tmp = vminf(array[15], array[45]);
    array[45] = vmaxf(array[15], array[45]);
    array[15] = tmp;
    tmp = vminf(array[18], array[48]);
    array[48] = vmaxf(array[18], array[48]);
    array[18] = tmp;
    tmp = vminf(array[19], array[49]);
    array[49] = vmaxf(array[19], array[49]);
    array[19] = tmp;
    tmp = vminf(array[22], array[52]);
    array[52] = vmaxf(array[22], array[52]);
    array[22] = tmp;
    tmp = vminf(array[23], array[53]);
    array[53] = vmaxf(array[23], array[53]);
    array[23] = tmp;
    tmp = vminf(array[26], array[56]);
    array[56] = vmaxf(array[26], array[56]);
    array[26] = tmp;
    tmp = vminf(array[27], array[57]);
    array[57] = vmaxf(array[27], array[57]);
    array[27] = tmp;
    tmp = vminf(array[30], array[60]);
    array[60] = vmaxf(array[30], array[60]);
    array[30] = tmp;
    tmp = vminf(array[31], array[61]);
    array[61] = vmaxf(array[31], array[61]);
    array[31] = tmp;
    tmp = vminf(array[34], array[64]);
    array[64] = vmaxf(array[34], array[64]);
    array[34] = tmp;
    tmp = vminf(array[35], array[65]);
    array[65] = vmaxf(array[35], array[65]);
    array[35] = tmp;
    tmp = vminf(array[38], array[68]);
    array[68] = vmaxf(array[38], array[68]);
    array[38] = tmp;
    tmp = vminf(array[39], array[69]);
    array[69] = vmaxf(array[39], array[69]);
    array[39] = tmp;
    tmp = vminf(array[42], array[72]);
    array[72] = vmaxf(array[42], array[72]);
    array[42] = tmp;
    tmp = vminf(array[43], array[73]);
    array[73] = vmaxf(array[43], array[73]);
    array[43] = tmp;
    tmp = vminf(array[46], array[76]);
    array[76] = vmaxf(array[46], array[76]);
    array[46] = tmp;
    tmp = vminf(array[47], array[77]);
    array[77] = vmaxf(array[47], array[77]);
    array[47] = tmp;
    tmp = vminf(array[50], array[80]);
    array[80] = vmaxf(array[50], array[80]);
    array[50] = tmp;
    tmp = vminf(array[2], array[16]);
    array[16] = vmaxf(array[2], array[16]);
    array[2] = tmp;
    tmp = vminf(array[3], array[17]);
    array[17] = vmaxf(array[3], array[17]);
    array[3] = tmp;
    tmp = vminf(array[6], array[20]);
    array[20] = vmaxf(array[6], array[20]);
    array[6] = tmp;
    tmp = vminf(array[7], array[21]);
    array[21] = vmaxf(array[7], array[21]);
    array[7] = tmp;
    tmp = vminf(array[10], array[24]);
    array[24] = vmaxf(array[10], array[24]);
    array[10] = tmp;
    tmp = vminf(array[11], array[25]);
    array[25] = vmaxf(array[11], array[25]);
    array[11] = tmp;
    tmp = vminf(array[14], array[28]);
    array[28] = vmaxf(array[14], array[28]);
    array[14] = tmp;
    tmp = vminf(array[15], array[29]);
    array[29] = vmaxf(array[15], array[29]);
    array[15] = tmp;
    tmp = vminf(array[18], array[32]);
    array[32] = vmaxf(array[18], array[32]);
    array[18] = tmp;
    tmp = vminf(array[19], array[33]);
    array[33] = vmaxf(array[19], array[33]);
    array[19] = tmp;
    tmp = vminf(array[22], array[36]);
    array[36] = vmaxf(array[22], array[36]);
    array[22] = tmp;
    tmp = vminf(array[23], array[37]);
    array[37] = vmaxf(array[23], array[37]);
    array[23] = tmp;
    tmp = vminf(array[26], array[40]);
    array[40] = vmaxf(array[26], array[40]);
    array[26] = tmp;
    tmp = vminf(array[27], array[41]);
    array[41] = vmaxf(array[27], array[41]);
    array[27] = tmp;
    tmp = vminf(array[30], array[44]);
    array[44] = vmaxf(array[30], array[44]);
    array[30] = tmp;
    tmp = vminf(array[31], array[45]);
    array[45] = vmaxf(array[31], array[45]);
    array[31] = tmp;
    tmp = vminf(array[34], array[48]);
    array[48] = vmaxf(array[34], array[48]);
    array[34] = tmp;
    tmp = vminf(array[35], array[49]);
    array[49] = vmaxf(array[35], array[49]);
    array[35] = tmp;
    tmp = vminf(array[38], array[52]);
    array[52] = vmaxf(array[38], array[52]);
    array[38] = tmp;
    tmp = vminf(array[39], array[53]);
    array[53] = vmaxf(array[39], array[53]);
    array[39] = tmp;
    tmp = vminf(array[42], array[56]);
    array[56] = vmaxf(array[42], array[56]);
    array[42] = tmp;
    tmp = vminf(array[43], array[57]);
    array[57] = vmaxf(array[43], array[57]);
    array[43] = tmp;
    tmp = vminf(array[46], array[60]);
    array[60] = vmaxf(array[46], array[60]);
    array[46] = tmp;
    tmp = vminf(array[47], array[61]);
    array[61] = vmaxf(array[47], array[61]);
    array[47] = tmp;
    tmp = vminf(array[50], array[64]);
    array[64] = vmaxf(array[50], array[64]);
    array[50] = tmp;
    tmp = vminf(array[51], array[65]);
    array[65] = vmaxf(array[51], array[65]);
    array[51] = tmp;
    tmp = vminf(array[54], array[68]);
    array[68] = vmaxf(array[54], array[68]);
    array[54] = tmp;
    tmp = vminf(array[55], array[69]);
    array[69] = vmaxf(array[55], array[69]);
    array[55] = tmp;
    tmp = vminf(array[58], array[72]);
    array[72] = vmaxf(array[58], array[72]);
    array[58] = tmp;
    tmp = vminf(array[59], array[73]);
    array[73] = vmaxf(array[59], array[73]);
    array[59] = tmp;
    tmp = vminf(array[62], array[76]);
    array[76] = vmaxf(array[62], array[76]);
    array[62] = tmp;
    tmp = vminf(array[63], array[77]);
    array[77] = vmaxf(array[63], array[77]);
    array[63] = tmp;
    tmp = vminf(array[66], array[80]);
    array[80] = vmaxf(array[66], array[80]);
    array[66] = tmp;
    tmp = vminf(array[2], array[8]);
    array[8] = vmaxf(array[2], array[8]);
    array[2] = tmp;
    tmp = vminf(array[3], array[9]);
    array[9] = vmaxf(array[3], array[9]);
    array[3] = tmp;
    tmp = vminf(array[6], array[12]);
    array[12] = vmaxf(array[6], array[12]);
    array[6] = tmp;
    tmp = vminf(array[7], array[13]);
    array[13] = vmaxf(array[7], array[13]);
    array[7] = tmp;
    tmp = vminf(array[10], array[16]);
    array[16] = vmaxf(array[10], array[16]);
    array[10] = tmp;
    tmp = vminf(array[11], array[17]);
    array[17] = vmaxf(array[11], array[17]);
    array[11] = tmp;
    tmp = vminf(array[14], array[20]);
    array[20] = vmaxf(array[14], array[20]);
    array[14] = tmp;
    tmp = vminf(array[15], array[21]);
    array[21] = vmaxf(array[15], array[21]);
    array[15] = tmp;
    tmp = vminf(array[18], array[24]);
    array[24] = vmaxf(array[18], array[24]);
    array[18] = tmp;
    tmp = vminf(array[19], array[25]);
    array[25] = vmaxf(array[19], array[25]);
    array[19] = tmp;
    tmp = vminf(array[22], array[28]);
    array[28] = vmaxf(array[22], array[28]);
    array[22] = tmp;
    tmp = vminf(array[23], array[29]);
    array[29] = vmaxf(array[23], array[29]);
    array[23] = tmp;
    tmp = vminf(array[26], array[32]);
    array[32] = vmaxf(array[26], array[32]);
    array[26] = tmp;
    tmp = vminf(array[27], array[33]);
    array[33] = vmaxf(array[27], array[33]);
    array[27] = tmp;
    tmp = vminf(array[30], array[36]);
    array[36] = vmaxf(array[30], array[36]);
    array[30] = tmp;
    tmp = vminf(array[31], array[37]);
    array[37] = vmaxf(array[31], array[37]);
    array[31] = tmp;
    tmp = vminf(array[34], array[40]);
    array[40] = vmaxf(array[34], array[40]);
    array[34] = tmp;
    tmp = vminf(array[35], array[41]);
    array[41] = vmaxf(array[35], array[41]);
    array[35] = tmp;
    tmp = vminf(array[38], array[44]);
    array[44] = vmaxf(array[38], array[44]);
    array[38] = tmp;
    tmp = vminf(array[39], array[45]);
    array[45] = vmaxf(array[39], array[45]);
    array[39] = tmp;
    tmp = vminf(array[42], array[48]);
    array[48] = vmaxf(array[42], array[48]);
    array[42] = tmp;
    tmp = vminf(array[43], array[49]);
    array[49] = vmaxf(array[43], array[49]);
    array[43] = tmp;
    tmp = vminf(array[46], array[52]);
    array[52] = vmaxf(array[46], array[52]);
    array[46] = tmp;
    tmp = vminf(array[47], array[53]);
    array[53] = vmaxf(array[47], array[53]);
    array[47] = tmp;
    tmp = vminf(array[50], array[56]);
    array[56] = vmaxf(array[50], array[56]);
    array[50] = tmp;
    tmp = vminf(array[51], array[57]);
    array[57] = vmaxf(array[51], array[57]);
    array[51] = tmp;
    tmp = vminf(array[54], array[60]);
    array[60] = vmaxf(array[54], array[60]);
    array[54] = tmp;
    tmp = vminf(array[55], array[61]);
    array[61] = vmaxf(array[55], array[61]);
    array[55] = tmp;
    tmp = vminf(array[58], array[64]);
    array[64] = vmaxf(array[58], array[64]);
    array[58] = tmp;
    tmp = vminf(array[59], array[65]);
    array[65] = vmaxf(array[59], array[65]);
    array[59] = tmp;
    tmp = vminf(array[62], array[68]);
    array[68] = vmaxf(array[62], array[68]);
    array[62] = tmp;
    tmp = vminf(array[63], array[69]);
    array[69] = vmaxf(array[63], array[69]);
    array[63] = tmp;
    tmp = vminf(array[66], array[72]);
    array[72] = vmaxf(array[66], array[72]);
    array[66] = tmp;
    tmp = vminf(array[67], array[73]);
    array[73] = vmaxf(array[67], array[73]);
    array[67] = tmp;
    tmp = vminf(array[70], array[76]);
    array[76] = vmaxf(array[70], array[76]);
    array[70] = tmp;
    tmp = vminf(array[71], array[77]);
    array[77] = vmaxf(array[71], array[77]);
    array[71] = tmp;
    tmp = vminf(array[74], array[80]);
    array[80] = vmaxf(array[74], array[80]);
    array[74] = tmp;
    tmp = vminf(array[2], array[4]);
    array[4] = vmaxf(array[2], array[4]);
    array[2] = tmp;
    tmp = vminf(array[3], array[5]);
    array[5] = vmaxf(array[3], array[5]);
    array[3] = tmp;
    tmp = vminf(array[6], array[8]);
    array[8] = vmaxf(array[6], array[8]);
    array[6] = tmp;
    tmp = vminf(array[7], array[9]);
    array[9] = vmaxf(array[7], array[9]);
    array[7] = tmp;
    tmp = vminf(array[10], array[12]);
    array[12] = vmaxf(array[10], array[12]);
    array[10] = tmp;
    tmp = vminf(array[11], array[13]);
    array[13] = vmaxf(array[11], array[13]);
    array[11] = tmp;
    tmp = vminf(array[14], array[16]);
    array[16] = vmaxf(array[14], array[16]);
    array[14] = tmp;
    tmp = vminf(array[15], array[17]);
    array[17] = vmaxf(array[15], array[17]);
    array[15] = tmp;
    tmp = vminf(array[18], array[20]);
    array[20] = vmaxf(array[18], array[20]);
    array[18] = tmp;
    tmp = vminf(array[19], array[21]);
    array[21] = vmaxf(array[19], array[21]);
    array[19] = tmp;
    tmp = vminf(array[22], array[24]);
    array[24] = vmaxf(array[22], array[24]);
    array[22] = tmp;
    tmp = vminf(array[23], array[25]);
    array[25] = vmaxf(array[23], array[25]);
    array[23] = tmp;
    tmp = vminf(array[26], array[28]);
    array[28] = vmaxf(array[26], array[28]);
    array[26] = tmp;
    tmp = vminf(array[27], array[29]);
    array[29] = vmaxf(array[27], array[29]);
    array[27] = tmp;
    tmp = vminf(array[30], array[32]);
    array[32] = vmaxf(array[30], array[32]);
    array[30] = tmp;
    tmp = vminf(array[31], array[33]);
    array[33] = vmaxf(array[31], array[33]);
    array[31] = tmp;
    tmp = vminf(array[34], array[36]);
    array[36] = vmaxf(array[34], array[36]);
    array[34] = tmp;
    tmp = vminf(array[35], array[37]);
    array[37] = vmaxf(array[35], array[37]);
    array[35] = tmp;
    tmp = vminf(array[38], array[40]);
    array[40] = vmaxf(array[38], array[40]);
    array[38] = tmp;
    tmp = vminf(array[39], array[41]);
    array[41] = vmaxf(array[39], array[41]);
    array[39] = tmp;
    tmp = vminf(array[42], array[44]);
    array[44] = vmaxf(array[42], array[44]);
    array[42] = tmp;
    tmp = vminf(array[43], array[45]);
    array[45] = vmaxf(array[43], array[45]);
    array[43] = tmp;
    tmp = vminf(array[46], array[48]);
    array[48] = vmaxf(array[46], array[48]);
    array[46] = tmp;
    tmp = vminf(array[47], array[49]);
    array[49] = vmaxf(array[47], array[49]);
    array[47] = tmp;
    tmp = vminf(array[50], array[52]);
    array[52] = vmaxf(array[50], array[52]);
    array[50] = tmp;
    tmp = vminf(array[51], array[53]);
    array[53] = vmaxf(array[51], array[53]);
    array[51] = tmp;
    tmp = vminf(array[54], array[56]);
    array[56] = vmaxf(array[54], array[56]);
    array[54] = tmp;
    tmp = vminf(array[55], array[57]);
    array[57] = vmaxf(array[55], array[57]);
    array[55] = tmp;
    tmp = vminf(array[58], array[60]);
    array[60] = vmaxf(array[58], array[60]);
    array[58] = tmp;
    tmp = vminf(array[59], array[61]);
    array[61] = vmaxf(array[59], array[61]);
    array[59] = tmp;
    tmp = vminf(array[62], array[64]);
    array[64] = vmaxf(array[62], array[64]);
    array[62] = tmp;
    tmp = vminf(array[63], array[65]);
    array[65] = vmaxf(array[63], array[65]);
    array[63] = tmp;
    tmp = vminf(array[66], array[68]);
    array[68] = vmaxf(array[66], array[68]);
    array[66] = tmp;
    tmp = vminf(array[67], array[69]);
    array[69] = vmaxf(array[67], array[69]);
    array[67] = tmp;
    tmp = vminf(array[70], array[72]);
    array[72] = vmaxf(array[70], array[72]);
    array[70] = tmp;
    tmp = vminf(array[71], array[73]);
    array[73] = vmaxf(array[71], array[73]);
    array[71] = tmp;
    tmp = vminf(array[74], array[76]);
    array[76] = vmaxf(array[74], array[76]);
    array[74] = tmp;
    tmp = vminf(array[75], array[77]);
    array[77] = vmaxf(array[75], array[77]);
    array[75] = tmp;
    tmp = vminf(array[78], array[80]);
    array[80] = vmaxf(array[78], array[80]);
    array[78] = tmp;
    array[1] = vmaxf(array[0], array[1]);
    array[3] = vmaxf(array[2], array[3]);
    array[5] = vmaxf(array[4], array[5]);
    array[7] = vmaxf(array[6], array[7]);
    array[9] = vmaxf(array[8], array[9]);
    array[11] = vmaxf(array[10], array[11]);
    array[13] = vmaxf(array[12], array[13]);
    array[15] = vmaxf(array[14], array[15]);
    array[17] = vmaxf(array[16], array[17]);
    array[19] = vmaxf(array[18], array[19]);
    array[21] = vmaxf(array[20], array[21]);
    array[23] = vmaxf(array[22], array[23]);
    array[25] = vmaxf(array[24], array[25]);
    array[27] = vmaxf(array[26], array[27]);
    array[29] = vmaxf(array[28], array[29]);
    array[31] = vmaxf(array[30], array[31]);
    array[33] = vmaxf(array[32], array[33]);
    array[35] = vmaxf(array[34], array[35]);
    array[37] = vmaxf(array[36], array[37]);
    array[39] = vmaxf(array[38], array[39]);
    array[40] = vminf(array[40], array[41]);
    array[42] = vminf(array[42], array[43]);
    array[44] = vminf(array[44], array[45]);
    array[46] = vminf(array[46], array[47]);
    array[48] = vminf(array[48], array[49]);
    array[50] = vminf(array[50], array[51]);
    array[52] = vminf(array[52], array[53]);
    array[54] = vminf(array[54], array[55]);
    array[56] = vminf(array[56], array[57]);
    array[58] = vminf(array[58], array[59]);
    array[60] = vminf(array[60], array[61]);
    array[62] = vminf(array[62], array[63]);
    array[64] = vminf(array[64], array[65]);
    array[66] = vminf(array[66], array[67]);
    array[68] = vminf(array[68], array[69]);
    array[70] = vminf(array[70], array[71]);
    array[72] = vminf(array[72], array[73]);
    array[74] = vminf(array[74], array[75]);
    array[76] = vminf(array[76], array[77]);
    array[78] = vminf(array[78], array[79]);
    array[64] = vmaxf(array[1], array[64]);
    array[66] = vmaxf(array[3], array[66]);
    array[68] = vmaxf(array[5], array[68]);
    array[70] = vmaxf(array[7], array[70]);
    array[9] = vminf(array[9], array[72]);
    array[11] = vminf(array[11], array[74]);
    array[13] = vminf(array[13], array[76]);
    array[15] = vminf(array[15], array[78]);
    array[17] = vminf(array[17], array[80]);
    array[40] = vmaxf(array[9], array[40]);
    array[42] = vmaxf(array[11], array[42]);
    array[44] = vmaxf(array[13], array[44]);
    array[46] = vmaxf(array[15], array[46]);
    array[48] = vmaxf(array[17], array[48]);
    array[50] = vmaxf(array[19], array[50]);
    array[52] = vmaxf(array[21], array[52]);
    array[54] = vmaxf(array[23], array[54]);
    array[25] = vminf(array[25], array[56]);
    array[27] = vminf(array[27], array[58]);
    array[29] = vminf(array[29], array[60]);
    array[31] = vminf(array[31], array[62]);
    array[33] = vminf(array[33], array[64]);
    array[35] = vminf(array[35], array[66]);
    array[37] = vminf(array[37], array[68]);
    array[39] = vminf(array[39], array[70]);
    array[40] = vmaxf(array[25], array[40]);
    array[42] = vmaxf(array[27], array[42]);
    array[44] = vmaxf(array[29], array[44]);
    array[46] = vmaxf(array[31], array[46]);
    array[33] = vminf(array[33], array[48]);
    array[35] = vminf(array[35], array[50]);
    array[37] = vminf(array[37], array[52]);
    array[39] = vminf(array[39], array[54]);
    array[40] = vmaxf(array[33], array[40]);
    array[42] = vmaxf(array[35], array[42]);
    array[37] = vminf(array[37], array[44]);
    array[39] = vminf(array[39], array[46]);
    array[40] = vmaxf(array[37], array[40]);
    array[39] = vminf(array[39], array[42]);
    return vmaxf(array[39], array[40]);
}
#endif

template<typename T, typename... ARGS>
inline T median(T arg, ARGS... args)
{
    return median(std::array<T, sizeof...(args) + 1>{std::move(arg), std::move(args)...});
}

template<typename T>
inline std::array<T, 4> middle4of6(const std::array<T, 6>& array)
{
    std::array<T, 4> res;

    res[0] = std::min(array[1], array[2]);
    res[1] = std::max(array[1], array[2]);
    T left = std::min(array[0], res[1]);
    res[1] = std::max(array[0], res[1]);
    T tmp = std::min(left, res[0]);
    res[0] = std::max(left, res[0]);
    left = tmp;
    res[3] = std::min(array[4], array[5]);
    T right = std::max(array[4], array[5]);
    res[2] = std::min(array[3], right);
    right = std::max(array[3], right);
    tmp = std::min(res[2], res[3]);
    res[3] = std::max(res[2], res[3]);
    res[2] = std::max(left, tmp);
    res[1] = std::min(res[1], right);

    return res;
}

#ifdef __SSE2__
template<>
inline std::array<vfloat, 4> middle4of6(const std::array<vfloat, 6>& array)
{
    std::array<vfloat, 4> res;

    res[0] = vminf(array[1], array[2]);
    res[1] = vmaxf(array[1], array[2]);
    vfloat left = vminf(array[0], res[1]);
    res[1] = vmaxf(array[0], res[1]);
    vfloat tmp = vminf(left, res[0]);
    res[0] = vmaxf(left, res[0]);
    left = tmp;
    res[3] = vminf(array[4], array[5]);
    vfloat right = vmaxf(array[4], array[5]);
    res[2] = vminf(array[3], right);
    right = vmaxf(array[3], right);
    tmp = vminf(res[2], res[3]);
    res[3] = vmaxf(res[2], res[3]);
    res[2] = vmaxf(left, tmp);
    res[1] = vminf(res[1], right);

    return res;
}
#endif

template<typename T>
inline std::array<T, 4> middle4of6(T arg0, T arg1, T arg2, T arg3, T arg4, T arg5)
{
    return middle4of6(std::array<T, 6>{std::move(arg0), std::move(arg1), std::move(arg2), std::move(arg3), std::move(arg4), std::move(arg5)});
}
