% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/repeated_measures_d.R
\name{repeated_measures_d}
\alias{repeated_measures_d}
\alias{rm_d}
\title{Standardized Mean Differences for Repeated Measures}
\usage{
repeated_measures_d(
  x,
  y,
  data = NULL,
  mu = 0,
  method = c("rm", "av", "z", "b", "d", "r"),
  adjust = TRUE,
  reference = NULL,
  ci = 0.95,
  alternative = "two.sided",
  verbose = TRUE,
  ...
)

rm_d(
  x,
  y,
  data = NULL,
  mu = 0,
  method = c("rm", "av", "z", "b", "d", "r"),
  adjust = TRUE,
  reference = NULL,
  ci = 0.95,
  alternative = "two.sided",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x, y}{Paired numeric vectors, or names of ones in \code{data}. \code{x} can also
be a formula:
\itemize{
\item \code{Pair(x,y) ~ 1} for wide data.
\item \code{y ~ condition | id} for long data, possibly with repetitions.
}}

\item{data}{An optional data frame containing the variables.}

\item{mu}{a number indicating the true value of the mean (or
    difference in means if you are performing a two sample test).}

\item{method}{Method of repeated measures standardized differences. See
details.}

\item{adjust}{Apply Hedges' small-sample bias correction? See \code{\link[=hedges_g]{hedges_g()}}.}

\item{reference}{(Optional) character value of the "group" used as the
reference. By default, the \emph{second} group is the reference group.}

\item{ci}{Confidence Interval (CI) level}

\item{alternative}{a character string specifying the alternative hypothesis;
Controls the type of CI returned: \code{"two.sided"} (default, two-sided CI),
\code{"greater"} or \code{"less"} (one-sided CI). Partial matching is allowed (e.g.,
\code{"g"}, \code{"l"}, \code{"two"}...). See \emph{One-Sided CIs} in \link{effectsize_CIs}.}

\item{verbose}{Toggle warnings and messages on or off.}

\item{...}{Arguments passed to or from other methods. When \code{x} is a formula,
these can be \code{subset} and \code{na.action}.}
}
\value{
A data frame with the effect size and their CIs (\code{CI_low} and
\code{CI_high}).
}
\description{
Compute effect size indices for standardized mean differences in repeated
measures data. Pair with any reported \code{stats::t.test(paired = TRUE)}.
\cr\cr
In a repeated-measures design, the same subjects are measured in multiple
conditions or time points. Unlike the case of independent groups, there are
multiple sources of variation that can be used to standardized the
differences between the means of the conditions / times.
}
\note{
\code{rm_d()} is an alias for \code{repeated_measures_d()}.
}
\section{Standardized Mean Differences for Repeated Measures}{
Unlike \link[=cohens_d]{Cohen's d} for independent groups, where standardization
naturally is done by the (pooled) population standard deviation (cf. Glass’s
\eqn{\Delta}), when measured across two conditions are dependent, there are
many more options for what error term to standardize by. Additionally, some
options allow for data to be replicated (many measurements per condition per
individual), others require a single observation per condition per individual
(aka, paired data; so replications are aggregated).

(It should be noted that all of these have awful and confusing notations.)

Standardize by...
\itemize{
\item \strong{Difference Score Variance: \eqn{d_{z}}} (\emph{Requires paired data}) - This
is akin to computing difference scores for each individual and then
computing a one-sample Cohen's \emph{d} (Cohen, 1988, pp. 48; see examples).
\item \strong{Within-Subject Variance: \eqn{d_{rm}}} (\emph{Requires paired data}) - Cohen
suggested adjusting \eqn{d_{z}} to estimate the "standard" between-subjects
\emph{d} by a factor of \eqn{\sqrt{2(1-r)}}, where \emph{r} is the Pearson correlation
between the paired measures (Cohen, 1988, pp. 48).
\item \strong{Control Variance: \eqn{d_{b}} (aka Becker's \emph{d})} (\emph{Requires paired
data}) - Standardized by the variance of the control condition (or in a pre-
post-treatment setting, the pre-treatment condition). This is akin to Glass'
\emph{delta} (\code{\link[=glass_delta]{glass_delta()}}) (Becker, 1988). Note that this is taken here as the
\emph{second} condition (\code{y}).
\item \strong{Average Variance: \eqn{d_{av}}} (\emph{Requires paired data}) - Instead of
standardizing by the variance in the of the control (or pre) condition,
Cumming suggests standardizing by the average variance of the two paired
conditions (Cumming, 2013, pp. 291).
\item \strong{All Variance: Just \eqn{d}} - This is the same as computing a standard
independent-groups Cohen's \emph{d} (Cohen, 1988). Note that CIs \emph{do} account for
the dependence, and so are typically more narrow (see examples).
\item \strong{Residual Variance: \eqn{d_{r}}} (\emph{Requires data with replications}) -
Divide by the pooled variance after all individual differences have been
partialled out (i.e., the residual/level-1 variance in an ANOVA or MLM
setting). In between-subjects designs where each subject contributes a single
response, this is equivalent to classical Cohen’s d. Priors in the
\code{BayesFactor} package are defined on this scale (Rouder et al., 2012).
\cr\cr
Note that for paired data, when the two conditions have equal variance,
\eqn{d_{rm}}, \eqn{d_{av}}, \eqn{d_{b}} are equal to \eqn{d}.
}
}

\section{Confidence (Compatibility) Intervals (CIs)}{
Confidence intervals are estimated using the standard normal parametric
method (see Algina & Keselman, 2003; Becker, 1988; Cooper et al., 2009;
Hedges & Olkin, 1985; Pustejovsky et al., 2014).
}

\section{CIs and Significance Tests}{
"Confidence intervals on measures of effect size convey all the information
in a hypothesis test, and more." (Steiger, 2004). Confidence (compatibility)
intervals and p values are complementary summaries of parameter uncertainty
given the observed data. A dichotomous hypothesis test could be performed
with either a CI or a p value. The 100 (1 - \eqn{\alpha})\% confidence
interval contains all of the parameter values for which \emph{p} > \eqn{\alpha}
for the current data and model. For example, a 95\% confidence interval
contains all of the values for which p > .05.
\cr\cr
Note that a confidence interval including 0 \emph{does not} indicate that the null
(no effect) is true. Rather, it suggests that the observed data together with
the model and its assumptions combined do not provided clear evidence against
a parameter value of 0 (same as with any other value in the interval), with
the level of this evidence defined by the chosen \eqn{\alpha} level (Rafi &
Greenland, 2020; Schweder & Hjort, 2016; Xie & Singh, 2013). To infer no
effect, additional judgments about what parameter values are "close enough"
to 0 to be negligible are needed ("equivalence testing"; Bauer & Kiesser,
1996).
}

\section{Plotting with \code{see}}{

The \code{see} package contains relevant plotting functions. See the \href{https://easystats.github.io/see/articles/effectsize.html}{plotting vignette in the \code{see} package}.
}

\examples{
# Paired data -------

data("sleep")
sleep2 <- reshape(sleep,
  direction = "wide",
  idvar = "ID", timevar = "group"
)

repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2)

# Same as:
# repeated_measures_d(sleep$extra[sleep$group==1],
#                     sleep$extra[sleep$group==2])
# repeated_measures_d(extra ~ group | ID, data = sleep)


# More options:
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, mu = -1)
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, alternative = "less")

# Other methods
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, method = "av")
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, method = "b")
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, method = "d")
repeated_measures_d(Pair(extra.1, extra.2) ~ 1, data = sleep2, method = "z", adjust = FALSE)

# d_z is the same as Cohen's d for one sample (of individual difference):
cohens_d(extra.1 - extra.2 ~ 1, data = sleep2)



# Repetition data -----------

data("rouder2016")

# For rm, ad, z, b, data is aggregated
repeated_measures_d(rt ~ cond | id, data = rouder2016)

# same as:
rouder2016_wide <- tapply(rouder2016[["rt"]], rouder2016[1:2], mean)
repeated_measures_d(rouder2016_wide[, 1], rouder2016_wide[, 2])

# For r or d, data is not aggragated:
repeated_measures_d(rt ~ cond | id, data = rouder2016, method = "r")
repeated_measures_d(rt ~ cond | id, data = rouder2016, method = "d", adjust = FALSE)

# d is the same as Cohen's d for two independent groups:
cohens_d(rt ~ cond, data = rouder2016, ci = NULL)

}
\references{
\itemize{
\item Algina, J., & Keselman, H. J. (2003). Approximate confidence intervals for
effect sizes. Educational and Psychological Measurement, 63(4), 537-553.
\item Becker, B. J. (1988). Synthesizing standardized mean‐change measures.
British Journal of Mathematical and Statistical Psychology, 41(2), 257-278.
\item Cohen, J. (1988). Statistical power analysis for the behavioral
sciences (2nd Ed.). New York: Routledge.
\item Cooper, H., Hedges, L., & Valentine, J. (2009). Handbook of research
synthesis and meta-analysis. Russell Sage Foundation, New York.
\item Cumming, G. (2013). Understanding the new statistics: Effect sizes,
confidence intervals, and meta-analysis. Routledge.
\item Hedges, L. V. & Olkin, I. (1985). Statistical methods for
meta-analysis. Orlando, FL: Academic Press.
\item Pustejovsky, J. E., Hedges, L. V., & Shadish, W. R. (2014).
Design-comparable effect sizes in multiple baseline designs: A general
modeling framework. Journal of Educational and Behavioral Statistics, 39(5),
368-393.
\item Rouder, J. N., Morey, R. D., Speckman, P. L., & Province, J. M. (2012).
Default Bayes factors for ANOVA designs. Journal of mathematical psychology,
56(5), 356-374.
}
}
\seealso{
\code{\link[=cohens_d]{cohens_d()}}, and \code{lmeInfo::g_mlm()} and \code{emmeans::effsize()} for
more flexible methods.

Other standardized differences: 
\code{\link{cohens_d}()},
\code{\link{mahalanobis_d}()},
\code{\link{means_ratio}()},
\code{\link{p_superiority}()},
\code{\link{rank_biserial}()}
}
\concept{standardized differences}
